/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.testing

import org.gradle.api.internal.tasks.testing.TestDescriptorInternal
import org.gradle.api.internal.tasks.testing.operations.ExecuteTestBuildOperationType
import org.gradle.integtests.fixtures.AbstractIntegrationSpec
import org.gradle.integtests.fixtures.BuildOperationsFixture
import org.gradle.integtests.fixtures.GroovyBuildScriptLanguage
import org.gradle.internal.operations.trace.BuildOperationRecord
import org.gradle.test.fixtures.file.TestFile

import java.time.Instant

class TestEventReporterIntegrationTest extends AbstractIntegrationSpec {
    def operations = new BuildOperationsFixture(executer, temporaryFolder)

    def "emits build operations for custom test"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("myTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdOut, "This is a test output on stdout")
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdErr, "This is a test output on stderr")
                                 myTest.succeeded(Instant.now())
                            }
                            try (def myTest = mySuite.reportTest("myTestInternal2", "My failing test :(")) {
                                 myTest.started(Instant.now())
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdErr, "Some text on stderr")
                                 myTest.failed(Instant.now(), "my failure")
                            }
                            try (def myTest = mySuite.reportTest("myTestInternal3", "another failing test")) {
                                 myTest.started(Instant.now())
                                 myTest.failed(Instant.now()) // no failure message
                            }
                            mySuite.failed(Instant.now())
                       }
                       reporter.failed(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        when:
        fails "customTest"

        then: "threw VerificationException"
        failure.assertHasCause("Test(s) failed.")

        def customTestOutput = failure.groupedOutput.task(":customTest")
        customTestOutput.assertOutputContains("""Custom test root > My Suite > My failing test :( FAILED
    my failure

Custom test root > My Suite > another failing test FAILED

3 tests completed, 1 succeeded, 2 failed""")

        then: "test build operations are emitted in expected hierarchy"
        def rootTestOp = operations.first(ExecuteTestBuildOperationType)
        def rootTestOpDetails = rootTestOp.details as Map<String, Map<String, ?>>
        (rootTestOpDetails.testDescriptor.name as String).startsWith("Custom test root")
        rootTestOpDetails.testDescriptor.className == null
        rootTestOpDetails.testDescriptor.composite

        def suiteTestOps = operations.children(rootTestOp, ExecuteTestBuildOperationType)
        suiteTestOps.size() == 1
        def suiteTestOpDetails = suiteTestOps[0].details as Map<String, Map<String, ?>>
        (suiteTestOpDetails.testDescriptor.name as String).startsWith("My Suite")
        suiteTestOpDetails.testDescriptor.className == "My Suite"
        suiteTestOpDetails.testDescriptor.composite

        def firstLevelTestOps = operations.children(suiteTestOps[0], ExecuteTestBuildOperationType).sort {
            (it.details as Map<String, TestDescriptorInternal>).testDescriptor.name
        }
        firstLevelTestOps.size() == 3
        def firstLevelTestOpDetails = firstLevelTestOps*.details as List<Map<String, Map<String, ?>>>
        firstLevelTestOpDetails*.testDescriptor.name == ["myTestInternal", "myTestInternal2", "myTestInternal3"]
        firstLevelTestOpDetails*.testDescriptor.displayName == ["My test!", "My failing test :(", "another failing test"]
        firstLevelTestOpDetails*.testDescriptor.className == ["My Suite", "My Suite", "My Suite"]
        firstLevelTestOpDetails*.testDescriptor.composite == [false, false, false]

        def firstTestOutputProgress = firstLevelTestOps[0].progress
        firstTestOutputProgress.size() == 2
        def firstTestOutputs = firstTestOutputProgress*.details.output as List<Map<String, ?>>
        firstTestOutputs[0].destination == "StdOut"
        firstTestOutputs[0].message == "This is a test output on stdout"
        firstTestOutputs[1].destination == "StdErr"
        firstTestOutputs[1].message == "This is a test output on stderr"

        def secondTestOutputProgress = firstLevelTestOps[1].progress
        secondTestOutputProgress.size() == 1
        def secondTestOutputs = secondTestOutputProgress[0].details.output as Map<String, ?>
        secondTestOutputs.destination == "StdErr"
        secondTestOutputs.message == "Some text on stderr"
    }

    def "use current time in start/finish events when tests emit ancient timestamps"() {
        given:
        def startTime = Instant.now()
        buildFile("""
            import java.time.Instant
            import javax.inject.Inject
            import java.time.temporal.ChronoUnit
            import java.time.temporal.TemporalUnit

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    def ancientTime = Instant.ofEpochMilli(1000)
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                        reporter.started(ancientTime)
                        try (def mySuite = reporter.reportTestGroup("My Suite")) {
                             mySuite.started(ancientTime.plusMillis(10))
                             try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                  myTest.started(ancientTime.plusMillis((20)))
                                  myTest.succeeded(ancientTime.plusMillis(30))
                             }
                             mySuite.succeeded(ancientTime.plusMillis(40))
                        }
                        reporter.succeeded(ancientTime.plusMillis(50))
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        when:
        succeeds "customTest"

        then:
        def rootTestOp = operations.first(ExecuteTestBuildOperationType)

        // just check that the timestamps are within our testing window
        Instant.ofEpochMilli(rootTestOp.startTime).isAfter(startTime)
        Instant.ofEpochMilli(rootTestOp.startTime).isBefore(Instant.now())

        Instant.ofEpochMilli(rootTestOp.endTime).isAfter(startTime)
        Instant.ofEpochMilli(rootTestOp.endTime).isBefore(Instant.now())

        // The result of the test execution has the reported timestamps
        rootTestOp.result.result.startTime == 1000
        rootTestOp.result.result.endTime == 1050
    }

    def "captures String metadata for custom test"() {
        given:
        singleCustomTestRecordingMetadata("my key", "'my value'")

        when:
        succeeds "customTest"

        then: "metadata is retrievable from build operations"
        List<BuildOperationRecord.Progress> testMetadata = getMetadataForOnlyTest()
        testMetadata.size() == 1
        def firstTestMetadataDetails = testMetadata*.details.metadata as List<Map<String, ?>>
        firstTestMetadataDetails.size() == 1
        firstTestMetadataDetails[0]["values"]["my key"] == "my value"
    }

    def "captures multiple metadata values for custom test"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(Instant.now(), "key1", "value1")
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdOut, "This is a test output on stdout")
                                 myTest.metadata(Instant.now(), "key2", "2")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        when:
        succeeds "customTest"

        then: "metadata is retrievable from build operations"
        List<BuildOperationRecord.Progress> testMetadata = getMetadataForOnlyTest()
        testMetadata.size() == 2
        def firstTestMetadataDetails = testMetadata*.details.metadata as List<Map<String, ?>>
        firstTestMetadataDetails.size() == 2
        firstTestMetadataDetails[0]["values"]["key1"] == "value1"
        firstTestMetadataDetails[1]["values"]["key2"] == "2"
    }

    def "captures multiple metadata values for multiple custom tests and correctly associates them"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(Instant.now(), "key1", "value1")
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdOut, "This is a test output on stdout")
                                 myTest.metadata(Instant.now(), "key2", "2")
                                 myTest.succeeded(Instant.now())
                            }

                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test 2!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(Instant.now(), "key3", "value4")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        when:
        succeeds "customTest"

        then: "metadata is retrievable from build operations"
        def rootTestOp = operations.first(ExecuteTestBuildOperationType)
        def rootTestOpDetails = rootTestOp.details as Map<String, Map<String, ?>>
        assert (rootTestOpDetails.testDescriptor.name as String).startsWith("Custom test root")

        def suiteTestOps = operations.children(rootTestOp, ExecuteTestBuildOperationType)
        assert suiteTestOps.size() == 1
        def suiteTestOpDetails = suiteTestOps[0].details as Map<String, Map<String, ?>>
        assert (suiteTestOpDetails.testDescriptor.name as String).startsWith("My Suite")

        def firstLevelTestOps = operations.children(suiteTestOps[0], ExecuteTestBuildOperationType).sort {
            (it.details as Map<String, TestDescriptorInternal>).testDescriptor.name
        }
        assert firstLevelTestOps.size() == 2
        def firstLevelTestOpDetails1 = firstLevelTestOps[0].details
        assert firstLevelTestOpDetails1.testDescriptor.name == "MyTestInternal"
        assert firstLevelTestOpDetails1.testDescriptor.displayName == "My test!"
        def firstLevelTest2OpDetails = firstLevelTestOps[1].details
        assert firstLevelTest2OpDetails.testDescriptor.name == "MyTestInternal"
        assert firstLevelTest2OpDetails.testDescriptor.displayName == "My test 2!"

        List<BuildOperationRecord.Progress> testMetadata1 = firstLevelTestOps[0].progress(ExecuteTestBuildOperationType.Metadata)
        testMetadata1.size() == 2
        def firstTestMetadataDetails = testMetadata1*.details.metadata as List<Map<String, ?>>
        firstTestMetadataDetails.size() == 2
        firstTestMetadataDetails[0]["values"]["key1"] == "value1"
        firstTestMetadataDetails[1]["values"]["key2"] == "2"

        List<BuildOperationRecord.Progress> testMetadata2 = firstLevelTestOps[1].progress(ExecuteTestBuildOperationType.Metadata)
        testMetadata2.size() == 1
        def secondTestMetadataDetails = testMetadata2*.details.metadata as List<Map<String, ?>>
        secondTestMetadataDetails.size() == 1
        secondTestMetadataDetails[0]["values"]["key3"] == "value4"
    }

    def "null metadata timestamps aren't allowed"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(null, "key", "value")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        expect:
        fails "customTest", "-S"

        and:
        failure.assertHasCause("logTime can not be null!")
    }

    def "null metadata keys aren't allowed"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(Instant.now(), null, "value")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        expect:
        fails "customTest"

        and:
        failure.assertHasCause("Metadata key can not be null!")
    }

    def "null metadata values aren't allowed"() {
        given:
        singleCustomTestRecordingMetadata("mykey", null)

        expect:
        fails "customTest"

        and:
        failure.assertHasCause("Metadata value can not be null!")
    }

    def "metadata events can reuse test start time"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 def start = Instant.now()
                                 myTest.started(start)
                                 myTest.metadata(start, "mykey", "myvalue")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        expect:
        succeeds "customTest"
    }

    def "metadata timestamps before test start time aren't validated"() {
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 def start = Instant.now()
                                 myTest.started(start)
                                 myTest.metadata(start.minusMillis(1), "mykey", "myvalue")
                                 def end = Instant.now()
                                 myTest.succeeded(end)
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        expect:
        succeeds "customTest"
    }

    def "metadata events can reuse keys, with last event reported"() {
        given:
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.metadata(Instant.now(), "mykey", "myvalue")
                                 myTest.metadata(Instant.now(), "mykey", "updated")
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)

        when:
        succeeds "customTest"

        then: "metadata is retrievable from build operations"
        List<BuildOperationRecord.Progress> testMetadata = getMetadataForOnlyTest()
        testMetadata.size() == 2
        def firstTestMetadataDetails = testMetadata*.details.metadata as List<Map<String, ?>>
        firstTestMetadataDetails.size() == 2
        firstTestMetadataDetails[0]["values"]["mykey"] == "myvalue"
        firstTestMetadataDetails[1]["values"]["mykey"] == "updated"
    }

    private TestFile singleCustomTestRecordingMetadata(String key, @GroovyBuildScriptLanguage String valueExpression) {
        buildFile("""
            import java.time.Instant

            abstract class CustomTestTask extends DefaultTask {
                @Inject
                abstract TestEventReporterFactory getTestEventReporterFactory()

                @Inject
                abstract ProjectLayout getLayout()

                @TaskAction
                void runTests() {
                    try (def reporter = testEventReporterFactory.createTestEventReporter(
                        "Custom test root",
                        getLayout().getBuildDirectory().dir("test-results/Custom test root").get(),
                        getLayout().getBuildDirectory().dir("reports/tests/Custom test root").get()
                    )) {
                       reporter.started(Instant.now())
                       try (def mySuite = reporter.reportTestGroup("My Suite")) {
                            mySuite.started(Instant.now())
                            try (def myTest = mySuite.reportTest("MyTestInternal", "My test!")) {
                                 myTest.started(Instant.now())
                                 myTest.output(Instant.now(), TestOutputEvent.Destination.StdOut, "This is a test output on stdout")
                                 myTest.metadata(Instant.now(), "$key", $valueExpression)
                                 myTest.succeeded(Instant.now())
                            }
                            mySuite.succeeded(Instant.now())
                       }
                       reporter.succeeded(Instant.now())
                   }
                }
            }

            tasks.register("customTest", CustomTestTask)
        """)
    }

    private List<BuildOperationRecord.Progress> getMetadataForOnlyTest() {
        def rootTestOp = operations.first(ExecuteTestBuildOperationType)
        def rootTestOpDetails = rootTestOp.details as Map<String, Map<String, ?>>
        assert (rootTestOpDetails.testDescriptor.name as String).startsWith("Custom test root")
        assert rootTestOpDetails.testDescriptor.className == null
        assert rootTestOpDetails.testDescriptor.composite

        def suiteTestOps = operations.children(rootTestOp, ExecuteTestBuildOperationType)
        assert suiteTestOps.size() == 1
        def suiteTestOpDetails = suiteTestOps[0].details as Map<String, Map<String, ?>>
        assert (suiteTestOpDetails.testDescriptor.name as String).startsWith("My Suite")
        assert suiteTestOpDetails.testDescriptor.className == "My Suite"
        assert suiteTestOpDetails.testDescriptor.composite

        def firstLevelTestOps = operations.children(suiteTestOps[0], ExecuteTestBuildOperationType).sort {
            (it.details as Map<String, TestDescriptorInternal>).testDescriptor.name
        }
        assert firstLevelTestOps.size() == 1
        def firstLevelTestOpDetails = firstLevelTestOps*.details as List<Map<String, Map<String, ?>>>
        assert firstLevelTestOpDetails*.testDescriptor.name == ["MyTestInternal"]
        assert firstLevelTestOpDetails*.testDescriptor.displayName == ["My test!"]
        assert firstLevelTestOpDetails*.testDescriptor.className == ["My Suite"]
        assert firstLevelTestOpDetails*.testDescriptor.composite == [false]

        return firstLevelTestOps[0].progress(ExecuteTestBuildOperationType.Metadata)
    }
}
