/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.reporting.dependencies.internal;

import com.googlecode.jatl.Html;
import org.gradle.api.Transformer;
import org.gradle.api.tasks.diagnostics.internal.ProjectDetails;
import org.gradle.reporting.HtmlPageBuilder;
import org.gradle.reporting.ReportRenderer;
import org.gradle.util.GradleVersion;

import java.io.Writer;
import java.util.Date;
import java.util.Set;

/**
 * Renders an index page for the HTML dependency report for a set of projects.
 */
public class ProjectsPageRenderer extends ReportRenderer<Set<ProjectDetails.ProjectNameAndPath>, HtmlPageBuilder<Writer>> {
    private final Transformer<String, ProjectDetails.ProjectNameAndPath> namingScheme;

    public ProjectsPageRenderer(Transformer<String, ProjectDetails.ProjectNameAndPath> namingScheme) {
        this.namingScheme = namingScheme;
    }

    @Override
    public void render(final Set<ProjectDetails.ProjectNameAndPath> projects, final HtmlPageBuilder<Writer> builder) {
        final String baseCssLink = builder.requireResource(getClass().getResource("/org/gradle/reporting/base-style.css"));
        final String cssLink = builder.requireResource(getClass().getResource("/org/gradle/api/tasks/diagnostics/htmldependencyreport/style.css"));

        new Html(builder.getOutput()) {{
            html();
                head();
                    meta().httpEquiv("Content-Type").content("text/html; charset=utf-8");
                    meta().httpEquiv("x-ua-compatible").content("IE=edge");
                    link().rel("stylesheet").type("text/css").href(baseCssLink).end();
                    link().rel("stylesheet").type("text/css").href(cssLink).end();
                    title().text("Dependency reports").end();
                end();
                body();
                    div().id("content");
                        h1().text("Dependency Reports").end();
                        div().classAttr("tab");
                            table();
                                thead();
                                    tr();
                                        th().text("Project").end();
                                        th().text("Description").end();
                                    end();
                                end();
                                tbody();
                                for (ProjectDetails.ProjectNameAndPath project : projects) {
                                    tr();
                                        td().a().href(namingScheme.transform(project)).text(project.getDisplayName()).end().end();
                                        td().text(project.getDescription()).end();
                                    end();
                                }
                                end();
                            end();
                        end();
                        div().id("footer");
                            p();
                                text("Generated by ");
                                a().href("https://www.gradle.org").text(GradleVersion.current().toString()).end();
                                text(" at " + builder.formatDate(new Date()));
                            end();
                        end();
                    end();
                end();
            endAll();
        }};
    }
}
