/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.locking;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.gradle.api.file.RegularFile;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.internal.DocumentationRegistry;
import org.gradle.api.internal.DomainObjectContext;
import org.gradle.api.internal.file.FileResolver;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.internal.locking.InvalidLockFileException;
import org.gradle.internal.resource.local.FileResourceListener;
import org.gradle.util.internal.GFileUtils;
import org.jspecify.annotations.Nullable;

public class LockFileReaderWriter {
    private static final Logger LOGGER = Logging.getLogger(LockFileReaderWriter.class);
    private static final String LIMITATIONS_DOC_LINK = new DocumentationRegistry().getDocumentationRecommendationFor("information on limitations", "dependency_locking", "locking_limitations");
    static final String FORMATTING_DOC_LINK = "Verify the lockfile content. " + new DocumentationRegistry().getDocumentationRecommendationFor("information on lock file format", "dependency_locking", "lock_state_location_and_format");
    static final String UNIQUE_LOCKFILE_NAME = "gradle.lockfile";
    static final String FILE_SUFFIX = ".lockfile";
    static final String DEPENDENCY_LOCKING_FOLDER = "gradle/dependency-locks";
    static final Charset CHARSET = StandardCharsets.UTF_8;
    static final List<String> LOCKFILE_HEADER_LIST = ImmutableList.of((Object)"# This is a Gradle generated file for dependency locking.", (Object)"# Manual edits can break the build and are not advised.", (Object)"# This file is expected to be part of source control.");
    static final String EMPTY_RESOLUTIONS_ENTRY = "empty=";
    static final String BUILD_SCRIPT_PREFIX = "buildscript-";
    static final String SETTINGS_SCRIPT_PREFIX = "settings-";
    private final Path lockFilesRoot;
    private final DomainObjectContext context;
    private final RegularFileProperty lockFile;
    private final FileResourceListener listener;

    public LockFileReaderWriter(FileResolver fileResolver, DomainObjectContext context, RegularFileProperty lockFile, FileResourceListener listener) {
        this.context = context;
        this.lockFile = lockFile;
        this.listener = listener;
        Path resolve = null;
        if (fileResolver.canResolveRelativePath()) {
            resolve = fileResolver.resolve((Object)DEPENDENCY_LOCKING_FOLDER).toPath();
            lockFile.set(fileResolver.resolve((Object)this.decorate(UNIQUE_LOCKFILE_NAME)));
        }
        this.lockFilesRoot = resolve;
        LOGGER.debug("Lockfiles root: {}", (Object)this.lockFilesRoot);
    }

    public @Nullable List<String> readLockFile(String lockId) {
        this.checkValidRoot();
        Path lockFile = this.lockFilesRoot.resolve(this.decorate(lockId) + FILE_SUFFIX);
        this.listener.fileObserved(lockFile.toFile());
        if (Files.exists(lockFile, new LinkOption[0])) {
            List<String> result;
            try {
                result = Files.readAllLines(lockFile, CHARSET);
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to load lock file", e);
            }
            List<String> lines = result;
            LockFileReaderWriter.filterNonModuleLines(lines);
            return lines;
        }
        return null;
    }

    private String decorate(String lockId) {
        if (this.context.isScript()) {
            if (this.context.isRootScript()) {
                return SETTINGS_SCRIPT_PREFIX + lockId;
            }
            return BUILD_SCRIPT_PREFIX + lockId;
        }
        return lockId;
    }

    private void checkValidRoot() {
        if (this.lockFilesRoot == null) {
            throw new IllegalStateException("Dependency locking cannot be used for " + this.context.getDisplayName() + ". " + LIMITATIONS_DOC_LINK);
        }
    }

    private static void filterNonModuleLines(List<String> lines) {
        Iterator<String> iterator = lines.iterator();
        while (iterator.hasNext()) {
            String value = iterator.next().trim();
            if (!value.startsWith("#") && !value.isEmpty()) continue;
            iterator.remove();
        }
    }

    public Map<String, List<String>> readUniqueLockFile() {
        this.checkValidRoot();
        Predicate<String> empty = String::isEmpty;
        Predicate<String> comment = s -> s.startsWith("#");
        Path uniqueLockFile = this.getUniqueLockfilePath();
        ArrayList emptyLockIds = new ArrayList();
        HashMap<String, List<String>> uniqueLockState = new HashMap<String, List<String>>(10);
        this.listener.fileObserved(uniqueLockFile.toFile());
        if (Files.exists(uniqueLockFile, new LinkOption[0])) {
            try (Stream<String> lines = Files.lines(uniqueLockFile, CHARSET);){
                lines.filter(empty.or(comment).negate()).filter(line -> {
                    if (line.startsWith(EMPTY_RESOLUTIONS_ENTRY)) {
                        LockFileReaderWriter.collectEmptyLockIds(line, emptyLockIds);
                        return false;
                    }
                    return true;
                }).forEach(line -> this.parseLine((String)line, (Map<String, List<String>>)uniqueLockState));
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to load unique lockfile", e);
            }
            for (String emptyLockId : emptyLockIds) {
                uniqueLockState.computeIfAbsent(emptyLockId, k -> new ArrayList());
            }
            return uniqueLockState;
        }
        return new HashMap<String, List<String>>();
    }

    private static void collectEmptyLockIds(String line, List<String> emptyLockIds) {
        if (line.length() > EMPTY_RESOLUTIONS_ENTRY.length()) {
            String[] lockIds = line.substring(EMPTY_RESOLUTIONS_ENTRY.length()).split(",");
            Collections.addAll(emptyLockIds, lockIds);
        }
    }

    private Path getUniqueLockfilePath() {
        return ((RegularFile)this.lockFile.get()).getAsFile().toPath();
    }

    private void parseLine(String line, Map<String, List<String>> result) {
        String[] lockIds;
        String[] split = line.split("=");
        if (split.length != 2) {
            throw new InvalidLockFileException("lock file specified in '" + this.getUniqueLockfilePath().toString() + "'. Line: '" + line + "'", FORMATTING_DOC_LINK);
        }
        for (String lockId : lockIds = split[1].split(",")) {
            result.compute(lockId, (k, v) -> {
                ArrayList<String> mapping = v == null ? new ArrayList<String>() : v;
                mapping.add(split[0]);
                return mapping;
            });
        }
    }

    public boolean canWrite() {
        return this.lockFilesRoot != null;
    }

    public void writeUniqueLockfile(Map<String, List<String>> lockState) {
        this.checkValidRoot();
        Path lockfilePath = this.getUniqueLockfilePath();
        if (lockState.isEmpty()) {
            GFileUtils.deleteQuietly((File)lockfilePath.toFile());
            return;
        }
        TreeMap<String, List<String>> dependencyToLockIds = new TreeMap<String, List<String>>();
        ArrayList<String> emptyLockIds = new ArrayList<String>();
        LockFileReaderWriter.mapLockStateFromDependencyToLockId(lockState, dependencyToLockIds, emptyLockIds);
        LockFileReaderWriter.writeUniqueLockfile(lockfilePath, dependencyToLockIds, emptyLockIds);
        this.cleanupLegacyLockFiles(lockState.keySet());
    }

    private void cleanupLegacyLockFiles(Set<String> lockIdsToDelete) {
        lockIdsToDelete.stream().map(f -> this.lockFilesRoot.resolve(this.decorate((String)f) + FILE_SUFFIX)).map(Path::toFile).forEach(GFileUtils::deleteQuietly);
    }

    private static void writeUniqueLockfile(Path lockfilePath, Map<String, List<String>> dependencyToLockId, List<String> emptyLockIds) {
        try {
            Files.createDirectories(lockfilePath.getParent(), new FileAttribute[0]);
            ArrayList<String> content = new ArrayList<String>(50);
            content.addAll(LOCKFILE_HEADER_LIST);
            for (Map.Entry<String, List<String>> entry : dependencyToLockId.entrySet()) {
                String builder = entry.getKey() + "=" + entry.getValue().stream().sorted().collect(Collectors.joining(","));
                content.add(builder);
            }
            content.add(EMPTY_RESOLUTIONS_ENTRY + emptyLockIds.stream().sorted().collect(Collectors.joining(",")));
            byte[] bytes = String.join((CharSequence)"\n", content).concat("\n").getBytes(CHARSET);
            Files.write(lockfilePath, bytes, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to write unique lockfile", e);
        }
    }

    private static void mapLockStateFromDependencyToLockId(Map<String, List<String>> lockState, Map<String, List<String>> dependencyToLockIds, List<String> emptyLockIds) {
        for (Map.Entry<String, List<String>> entry : lockState.entrySet()) {
            List<String> dependencies = entry.getValue();
            if (dependencies.isEmpty()) {
                emptyLockIds.add(entry.getKey());
                continue;
            }
            for (String dependency : dependencies) {
                dependencyToLockIds.compute(dependency, (k, v) -> {
                    ArrayList<String> confs = v;
                    if (v == null) {
                        confs = new ArrayList<String>();
                    }
                    confs.add((String)entry.getKey());
                    return confs;
                });
            }
        }
    }
}

