/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.tasks.testing;

import org.gradle.api.tasks.testing.TestFailure;
import org.gradle.api.tasks.testing.TestMetadataEvent;
import org.gradle.api.tasks.testing.TestOutputEvent;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.time.Instant;
import java.util.List;
import java.util.Map;

@NullMarked
class LifecycleTrackingTestEventReporter<T extends TestEventReporterInternal> implements TestEventReporterInternal {
    protected final T delegate;

    private enum State {
        CREATED, STARTED, COMPLETED, CLOSED;
    }
    private State state = State.CREATED;

    LifecycleTrackingTestEventReporter(T delegate) {
        this.delegate = delegate;
    }

    @Override
    public void started(Instant startTime) {
        if (state != State.CREATED) {
            throw new IllegalStateException("started(...) cannot be called twice");
        }
        state = State.STARTED;
        delegate.started(startTime);
    }

    @Override
    public void output(Instant logTime, TestOutputEvent.Destination destination, String output) {
        requireRunning();
        delegate.output(logTime, destination, output);
    }

    @Override
    public void metadata(Instant logTime, String key, String value) {
        requireRunning();
        delegate.metadata(logTime, key, value);
    }

    @Override
    public void metadata(Instant logTime, Map<String, String> values) {
        requireRunning();
        delegate.metadata(logTime, values);
    }

    @Override
    public void metadata(TestMetadataEvent metadataEvent) {
        requireRunning();
        delegate.metadata(metadataEvent);
    }

    @Override
    public void succeeded(Instant endTime) {
        markCompleted();
        delegate.succeeded(endTime);
    }

    @Override
    public void skipped(Instant endTime) {
        markCompleted();
        delegate.skipped(endTime);
    }

    @Override
    public void skipped(Instant endTime, @Nullable TestFailure assumptionFailure) {
        markCompleted();
        delegate.skipped(endTime, assumptionFailure);
    }

    @Override
    public void failed(Instant endTime, String message, String additionalContent) {
        markCompleted();
        delegate.failed(endTime, message, additionalContent);
    }

    @Override
    public void failed(Instant endTime, List<TestFailure> failures) {
        markCompleted();
        delegate.failed(endTime, failures);
    }

    @Override
    public void close() {
        if (state == State.CLOSED) {
            return;
        }

        delegate.close();

        if (state == State.STARTED) {
            throw new IllegalStateException("succeeded(...)/skipped(...)/failed(...) must be called before close() if started(...) was called");
        }
        state = State.CLOSED;
    }

    protected boolean isCompleted() {
        return state == State.CLOSED;
    }

    protected void requireRunning() {
        switch (state) {
            case CREATED:
                throw new IllegalStateException("started(...) must be called before any other method");
            case COMPLETED:
                throw new IllegalStateException("succeeded(...)/skipped(...)/failed(...) has already been called");
            case CLOSED:
                throw new IllegalStateException("close() has already been called");
            case STARTED:
                break;
        }
    }

    protected void markCompleted() {
        requireRunning();
        state = State.COMPLETED;
    }

    @Override
    public String toString() {
        return delegate.toString();
    }
}
