/*
 * Copyright 2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.plugins.antlr

import org.apache.commons.lang3.StringUtils
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.features.file.ProjectFeatureLayout
import org.gradle.api.internal.plugins.BindsProjectFeature
import org.gradle.api.internal.plugins.ProjectFeatureBindingBuilder
import org.gradle.api.internal.plugins.ProjectFeatureBinding
import org.gradle.api.internal.plugins.features.dsl.bindProjectFeatureToBuildModel
import org.gradle.api.model.ObjectFactory
import org.gradle.api.plugins.antlr.internal.DefaultAntlrSourceDirectorySet
import org.gradle.api.plugins.java.JavaClasses
import org.gradle.features.registration.TaskRegistrar
import org.gradle.language.base.plugins.LifecycleBasePlugin
import javax.inject.Inject

@BindsProjectFeature(AntlrProjectFeaturePlugin.Binding::class)
class AntlrProjectFeaturePlugin : Plugin<Project> {
    /**
     * javaLibrary {
     *     sources {
     *         javaSources("main") {
     *             antlr {
     *             }
     *         }
     *     }
     * }
     */
    class Binding : ProjectFeatureBinding {
        override fun bind(builder: ProjectFeatureBindingBuilder) {
            builder.bindProjectFeatureToBuildModel(
                "antlr",
                AntlrGrammarsDefinition::class,
                JavaClasses::class
            ) { definition, buildModel, target ->
                val services = objectFactory.newInstance(Services::class.java)
                val parentModel = getBuildModel(target)

                definition.grammarSources = createAntlrSourceDirectorySet(parentModel.name, objectFactory)
                val outputDirectory = services.projectLayout.contextBuildDirectory.map { buildDir -> buildDir.dir("/generated-src/antlr/" + definition.grammarSources.getName()) }

                // Add the generated antlr sources to the java sources
                parentModel.inputSources.srcDir(outputDirectory)

                services.taskRegistrar.register("generate" + StringUtils.capitalize(parentModel.name) + "AntlrSources", AntlrTask::class.java) { antlrTask ->
                    antlrTask.group = LifecycleBasePlugin.BUILD_GROUP
                    antlrTask.description = "Generates sources from the " + definition.grammarSources.name + " Antlr grammars."
                    antlrTask.source = definition.grammarSources
                    antlrTask.outputDirectory = outputDirectory.get().asFile
                }

                buildModel.generatedSourcesDir.set(outputDirectory)
            }
        }

        private fun createAntlrSourceDirectorySet(parentDisplayName: String, objectFactory: ObjectFactory): AntlrSourceDirectorySet {
            val name = "$parentDisplayName.antlr"
            val displayName = "$parentDisplayName Antlr source"
            val antlrSourceSet: AntlrSourceDirectorySet = objectFactory.newInstance(DefaultAntlrSourceDirectorySet::class.java, objectFactory.sourceDirectorySet(name, displayName))
            antlrSourceSet.filter.include("**/*.g")
            antlrSourceSet.filter.include("**/*.g4")
            return antlrSourceSet
        }

        interface Services {
            @get:Inject
            val taskRegistrar: TaskRegistrar

            @get:Inject
            val projectLayout: ProjectFeatureLayout
        }
    }

    override fun apply(target: Project) = Unit
}
