/*
 * Copyright 2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.integtests.tooling.r940

import org.gradle.integtests.fixtures.TestResources
import org.gradle.integtests.tooling.fixture.TextUtil
import org.gradle.integtests.tooling.fixture.ToolingApiSpecification
import org.gradle.test.fixtures.file.TestDirectoryProvider
import org.gradle.test.fixtures.file.TestNameTestDirectoryProvider
import testengines.TestEnginesFixture

abstract class AbstractResourceBasedTestingCrossVersionTest extends ToolingApiSpecification {

    private static final String ENGINE_COPY_TO_DIR_NAME = "test-engine-build"

    private static File engineJarLib
    private static File engineBuildDir
    private static File perClassDir

    abstract List<TestEnginesFixture.TestEngines> getEnginesToSetup()

    def setup() {
        if (!engineJarLib) {
            TestDirectoryProvider testClassDirectoryProvider = new TestNameTestDirectoryProvider.UniquePerTestClassDirectoryProvider(this.getClass())
            TestResources resources = new TestResources(testClassDirectoryProvider, TestEnginesFixture.TestEngines.class, this.getClass())

            perClassDir = testClassDirectoryProvider.getTestDirectory().getParentFile()

            resources.maybeCopy("shared")
            enginesToSetup.forEach {
                resources.maybeCopy(it.name)
            }

            engineBuildDir = testClassDirectoryProvider.testDirectory.file(ENGINE_COPY_TO_DIR_NAME)

            withConnection(connector().forProjectDirectory(engineBuildDir)) {
                it.newBuild().forTasks("build").run()
            }
            engineJarLib = engineBuildDir.file("build/libs/${ENGINE_COPY_TO_DIR_NAME}.jar")
        }
    }

    protected String enableEngineForSuite() {
        return """
                useJUnitJupiter()

                dependencies {
                    implementation files("${TextUtil.normaliseFileSeparators(engineJarLib.absolutePath)}")
                }
        """
    }

    protected void writeTestClasses() {
        file("src/test/java/SomeTest.java") << """
            import org.junit.jupiter.api.Test;

            public class SomeTest {
                @Test
                public void testMethod() {
                    System.out.println("Tested!");
                }
            }
        """
    }

    protected void writeTestDefinitions(String path = "src/test/definitions") {
        file("$path/SomeTestSpec.rbt") << """<?xml version="1.0" encoding="UTF-8" ?>
            <tests>
                <test name="foo" />
                <test name="bar" />
            </tests>
        """
        file("$path/subSomeOtherTestSpec.rbt") << """<?xml version="1.0" encoding="UTF-8" ?>
            <tests>
                <test name="other" />
            </tests>
        """
    }
}
