/*
 * Copyright 2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gradle.api.internal.classpath;

import org.gradle.internal.classpath.ClassPath;
import org.gradle.internal.classpath.DefaultClassPath;
import org.gradle.internal.installation.GradleInstallation;
import org.gradle.util.internal.GUtil;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * Determines the classpath for a module by looking for a '${module}-classpath.properties' resource with 'name' set to the name of the module.
 */
@NullMarked
public class DefaultModuleRegistry implements ModuleRegistry {

    private final GradleInstallation gradleInstallation;
    private final Map<String, Module> modules = new HashMap<>();
    private final Map<String, Module> externalModules = new HashMap<>();

    public DefaultModuleRegistry(@Nullable GradleInstallation gradleInstallation) {
        if (gradleInstallation == null) {
            throw new IllegalArgumentException("A Gradle installation is required to execute Gradle.");
        }

        this.gradleInstallation = gradleInstallation;
    }

    @Override
    public Module getExternalModule(String name) {
        Module module = externalModules.get(name);
        if (module == null) {
            module = loadExternalModule(name);
            externalModules.put(name, module);
        }
        return module;
    }

    private Module loadExternalModule(String name) {
        File externalJar = findJar(name);
        if (externalJar == null) {
            throw new UnknownModuleException(String.format("Cannot locate JAR for module '%s' in distribution directory '%s'.", name, gradleInstallation.getGradleHome()));
        }
        return new DefaultModule(name, Collections.singleton(externalJar), Collections.emptySet());
    }

    @Override
    public Module getModule(String name) {
        Module module = modules.get(name);
        if (module == null) {
            module = loadModule(name);
            modules.put(name, module);
        }
        return module;
    }

    @Override
    @Nullable
    public Module findModule(String name) {
        Module module = modules.get(name);
        if (module == null) {
            module = loadOptionalModule(name);
            if (module != null) {
                modules.put(name, module);
            }
        }
        return module;
    }

    private Module loadModule(String moduleName) {
        Module module = loadOptionalModule(moduleName);
        if (module != null) {
            return module;
        }
        throw new UnknownModuleException(String.format("Cannot locate JAR for module '%s' in distribution directory '%s'.", moduleName, gradleInstallation.getGradleHome()));
    }

    private @Nullable Module loadOptionalModule(final String moduleName) {
        File jarFile = findJar(moduleName);
        if (jarFile != null) {
            Set<File> implementationClasspath = new LinkedHashSet<>();
            implementationClasspath.add(jarFile);
            Properties properties = loadModuleProperties(moduleName, jarFile);
            return module(moduleName, properties, implementationClasspath);
        }
        return null;
    }

    private Module module(String moduleName, Properties properties, Set<File> implementationClasspath) {
        String[] runtimeJarNames = split(properties.getProperty("runtime"));
        Set<File> runtimeClasspath = findDependencyJars(moduleName, runtimeJarNames);

        String[] projects = split(properties.getProperty("projects"));
        String[] optionalProjects = split(properties.getProperty("optional"));
        return new DefaultModule(moduleName, implementationClasspath, runtimeClasspath, projects, optionalProjects);
    }

    private Set<File> findDependencyJars(String moduleName, String[] jarNames) {
        Set<File> runtimeClasspath = new LinkedHashSet<>();
        for (String jarName : jarNames) {
            runtimeClasspath.add(findDependencyJar(moduleName, jarName));
        }
        return runtimeClasspath;
    }

    private Set<Module> getModules(String[] projectNames) {
        Set<Module> modules = new LinkedHashSet<>();
        for (String project : projectNames) {
            modules.add(getModule(project));
        }
        return modules;
    }

    private String[] split(@Nullable String value) {
        if (value == null) {
            return new String[0];
        }
        value = value.trim();
        if (value.length() == 0) {
            return new String[0];
        }
        return value.split(",");
    }

    private static Properties loadModuleProperties(String name, File jarFile) {
        try (ZipFile zipFile = new ZipFile(jarFile)) {
            String entryName = name + "-classpath.properties";
            ZipEntry entry = zipFile.getEntry(entryName);
            if (entry == null) {
                throw new IllegalStateException("Did not find " + entryName + " in " + jarFile.getAbsolutePath());
            }
            try (InputStream is = zipFile.getInputStream(entry)) {
                return GUtil.loadProperties(is);
            }
        } catch (IOException e) {
            throw new UncheckedIOException(String.format("Could not load properties for module '%s' from %s", name, jarFile), e);
        }
    }

    private @Nullable File findJar(String name) {
        Pattern pattern = Pattern.compile(Pattern.quote(name) + "-\\d.*\\.jar");
        for (File libDir : gradleInstallation.getLibDirs()) {
            File[] files = libDir.listFiles();
            if (files != null) {
                for (File file : files) {
                    if (pattern.matcher(file.getName()).matches()) {
                        return file;
                    }
                }
            }
        }
        return null;
    }

    private File findDependencyJar(String module, String name) {
        for (File libDir : gradleInstallation.getLibDirs()) {
            File jarFile = new File(libDir, name);
            if (jarFile.isFile()) {
                return jarFile;
            }
        }
        throw new IllegalArgumentException(String.format("Cannot find JAR '%s' required by module '%s' using classpath or distribution directory '%s'", name, module, gradleInstallation.getGradleHome()));
    }

    private class DefaultModule implements Module {

        private final String name;
        private final String[] projects;
        private final String[] optionalProjects;
        private final ClassPath implementationClasspath;
        private final ClassPath runtimeClasspath;
        private final ClassPath classpath;

        public DefaultModule(String name, Set<File> implementationClasspath, Set<File> runtimeClasspath, String[] projects, String[] optionalProjects) {
            this.name = name;
            this.projects = projects;
            this.optionalProjects = optionalProjects;
            this.implementationClasspath = DefaultClassPath.of(implementationClasspath);
            this.runtimeClasspath = DefaultClassPath.of(runtimeClasspath);
            Set<File> classpath = new LinkedHashSet<>();
            classpath.addAll(implementationClasspath);
            classpath.addAll(runtimeClasspath);
            this.classpath = DefaultClassPath.of(classpath);
        }

        public DefaultModule(String name, Set<File> singleton, Set<File> files) {
            this(name, singleton, files, NO_PROJECTS, NO_PROJECTS);
        }

        @Override
        public String toString() {
            return "module '" + name + "'";
        }

        @Override
        public Set<Module> getRequiredModules() {
            return getModules(projects);
        }

        @Override
        public ClassPath getImplementationClasspath() {
            return implementationClasspath;
        }

        @Override
        public ClassPath getRuntimeClasspath() {
            return runtimeClasspath;
        }

        @Override
        public ClassPath getClasspath() {
            return classpath;
        }

        @Override
        public Set<Module> getAllRequiredModules() {
            Set<Module> modules = new LinkedHashSet<>();
            collectRequiredModules(modules);
            return modules;
        }

        @Override
        public ClassPath getAllRequiredModulesClasspath() {
            ClassPath classPath = ClassPath.EMPTY;
            for (Module module : getAllRequiredModules()) {
                classPath = classPath.plus(module.getClasspath());
            }
            return classPath;
        }

        private void collectRequiredModules(Set<Module> modules) {
            if (!modules.add(this)) {
                return;
            }
            for (Module module : getRequiredModules()) {
                collectDependenciesOf(module, modules);
            }
            for (String optionalProject : optionalProjects) {
                Module module = findModule(optionalProject);
                if (module != null) {
                    collectDependenciesOf(module, modules);
                }
            }
        }

        private void collectDependenciesOf(Module module, Set<Module> modules) {
            ((DefaultModule) module).collectRequiredModules(modules);
        }

        private @Nullable Module findModule(String optionalProject) {
            try {
                return getModule(optionalProject);
            } catch (UnknownModuleException ex) {
                return null;
            }
        }
    }

    private static final String[] NO_PROJECTS = new String[0];
}
