/*
 * Copyright 2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.internal.buildtree.control

import org.gradle.api.GradleException
import org.gradle.api.logging.Logging
import org.gradle.internal.buildtree.BuildActionModelRequirements
import org.gradle.internal.buildtree.BuildModelParameters
import org.gradle.internal.buildtree.BuildModelParametersFactory
import org.gradle.internal.cc.buildtree.BuildModelParametersProvider
import org.gradle.util.internal.IncubationLogger

internal class DefaultBuildModelParametersFactory : BuildModelParametersFactory {

    private val logger = Logging.getLogger(DefaultBuildModelParametersFactory::class.java)

    override fun parametersForRootBuildTree(requirements: BuildActionModelRequirements): BuildModelParameters {
        val startParameter = requirements.startParameter

        // Isolated projects also implies configuration cache
        if (startParameter.isolatedProjects.get() && !startParameter.configurationCache.get()) {
            if (startParameter.configurationCache.isExplicit) {
                throw GradleException("The configuration cache cannot be disabled when isolated projects is enabled.")
            }
        }

        val modelParameters = BuildModelParametersProvider.parameters(requirements)
        logger.info("Operational build model parameters: {}", modelParameters.toDisplayMap())

        modelParameters.configurationCacheDisabledReason?.let { reason ->
            logger.lifecycle("{} as configuration cache cannot be reused {}", requirements.actionDisplayName.capitalizedDisplayName, reason)
        }

        if (modelParameters.isIsolatedProjects) {
            IncubationLogger.incubatingFeatureUsed("Isolated projects")
        } else {
            if (modelParameters.isConfigurationCacheParallelStore) {
                IncubationLogger.incubatingFeatureUsed("Parallel Configuration Cache")
            }
            if (modelParameters.isConfigureOnDemand) {
                IncubationLogger.incubatingFeatureUsed("Configuration on demand")
            }
        }

        return modelParameters
    }

    override fun parametersForNestedBuildTree(requirements: BuildActionModelRequirements): BuildModelParameters {
        return BuildModelParametersProvider.parametersForNestedBuildTree(requirements.startParameter)
    }
}
