/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.launcher.exec;

import org.gradle.internal.UncheckedException;
import org.gradle.internal.build.BuildLayoutValidator;
import org.gradle.internal.buildtree.BuildActionModelRequirements;
import org.gradle.internal.buildtree.BuildActionRunner;
import org.gradle.internal.buildtree.BuildModelParameters;
import org.gradle.internal.buildtree.BuildModelParametersFactory;
import org.gradle.internal.buildtree.BuildTreeState;
import org.gradle.internal.buildtree.RunTasksRequirements;
import org.gradle.internal.hash.HashCode;
import org.gradle.internal.hash.Hashing;
import org.gradle.internal.id.UniqueId;
import org.gradle.internal.invocation.BuildAction;
import org.gradle.internal.scopeids.id.BuildInvocationScopeId;
import org.gradle.internal.service.ServiceRegistry;
import org.gradle.internal.session.BuildSessionActionExecutor;
import org.gradle.internal.session.BuildSessionContext;
import org.gradle.internal.snapshot.ValueSnapshotter;
import org.gradle.tooling.internal.provider.action.BuildModelAction;
import org.gradle.tooling.internal.provider.action.ClientProvidedBuildAction;
import org.gradle.tooling.internal.provider.action.ClientProvidedPhasedAction;
import org.gradle.tooling.internal.provider.serialization.SerializedPayload;

import java.util.function.Supplier;

/**
 * A {@link BuildActionExecutor} responsible for establishing the build tree for a single invocation of a {@link BuildAction}.
 */
public class BuildTreeLifecycleBuildActionExecutor implements BuildSessionActionExecutor {

    private final BuildModelParametersFactory buildModelParametersFactory;
    private final BuildLayoutValidator buildLayoutValidator;
    private final ValueSnapshotter valueSnapshotter;

    public BuildTreeLifecycleBuildActionExecutor(
        BuildModelParametersFactory modelParametersFactory,
        BuildLayoutValidator buildLayoutValidator,
        ValueSnapshotter valueSnapshotter
    ) {
        this.buildModelParametersFactory = modelParametersFactory;
        this.buildLayoutValidator = buildLayoutValidator;
        this.valueSnapshotter = valueSnapshotter;
    }

    @Override
    public BuildActionRunner.Result execute(BuildAction action, BuildSessionContext buildSession) {
        BuildActionRunner.Result result = null;
        try {
            buildLayoutValidator.validate(action.getStartParameter());

            BuildActionModelRequirements actionRequirements = buildActionModelRequirementsFor(action);
            BuildModelParameters buildModelParameters = buildModelParametersFactory.parametersForRootBuildTree(actionRequirements);
            result = runRootBuildAction(action, buildSession.getServices(), actionRequirements, buildModelParameters);
        } catch (Throwable t) {
            if (result == null) {
                // Did not create a result
                // Note: throw the failure rather than returning a result object containing the failure, as console failure logging based on the _result_ happens down in the root build scope
                // whereas console failure logging based on the _thrown exception_ happens up outside session scope. It would be better to refactor so that a result can be returned from here
                throw UncheckedException.throwAsUncheckedException(t);
            } else {
                // Cleanup has failed, combine the cleanup failure with other failures that may be packed in the result
                // Note: throw the failure rather than returning a result object containing the failure, as console failure logging based on the _result_ happens down in the root build scope
                // whereas console failure logging based on the _thrown exception_ happens up outside session scope. It would be better to refactor so that a result can be returned from here
                throw UncheckedException.throwAsUncheckedException(result.addFailure(t).getBuildFailure());
            }
        }
        return result;
    }

    /**
     * Creates a new build tree and runs the action on behalf of the root build in that tree.
     * <p>
     * The build tree and its services are disposed of before this method returns.
     */
    private static BuildActionRunner.Result runRootBuildAction(
        BuildAction action,
        ServiceRegistry buildSessionServices,
        BuildActionModelRequirements buildActionRequirements,
        BuildModelParameters buildModelParameters
    ) {
        BuildInvocationScopeId buildInvocationScopeId = new BuildInvocationScopeId(UniqueId.generate());
        try (BuildTreeState buildTree = new BuildTreeState(buildSessionServices, buildActionRequirements, buildModelParameters, buildInvocationScopeId)) {
            return buildTree.getServices().get(RootBuildLifecycleBuildActionExecutor.class).execute(action);
        }
    }

    private BuildActionModelRequirements buildActionModelRequirementsFor(BuildAction action) {
        if (action instanceof BuildModelAction && action.isCreateModel()) {
            BuildModelAction buildModelAction = (BuildModelAction) action;
            Object payload = buildModelAction.getModelName();
            return new QueryModelRequirements(action.getStartParameter(), action.isRunTasks(), payloadHashProvider(payload));
        } else if (action instanceof ClientProvidedBuildAction) {
            SerializedPayload payload = ((ClientProvidedBuildAction) action).getAction();
            return new RunActionRequirements(action.getStartParameter(), action.isRunTasks(), payloadHashProvider(payload));
        } else if (action instanceof ClientProvidedPhasedAction) {
            SerializedPayload payload = ((ClientProvidedPhasedAction) action).getPhasedAction();
            return new RunPhasedActionRequirements(action.getStartParameter(), action.isRunTasks(), payloadHashProvider(payload));
        } else {
            return new RunTasksRequirements(action.getStartParameter());
        }
    }

    private Supplier<HashCode> payloadHashProvider(Object payload) {
        ValueSnapshotter valueSnapshotter = this.valueSnapshotter;
        return () -> Hashing.hashHashable(valueSnapshotter.snapshot(payload));
    }
}
