/*
 * Copyright 2013 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.plugins.ide.internal.tooling;

import org.gradle.api.GradleException;
import org.gradle.api.initialization.ProjectDescriptor;
import org.gradle.api.internal.GradleInternal;
import org.gradle.api.internal.SettingsInternal;
import org.gradle.internal.build.BuildState;
import org.gradle.internal.build.BuildStateRegistry;
import org.gradle.internal.build.IncludedBuildState;
import org.gradle.internal.build.RootBuildState;
import org.gradle.internal.composite.BuildIncludeListener;
import org.gradle.internal.composite.IncludedBuildInternal;
import org.gradle.internal.problems.failure.Failure;
import org.gradle.internal.problems.failure.FailureFactory;
import org.gradle.plugins.ide.internal.tooling.model.BasicGradleProject;
import org.gradle.plugins.ide.internal.tooling.model.DefaultGradleBuild;
import org.gradle.tooling.internal.gradle.DefaultBuildIdentifier;
import org.gradle.tooling.internal.gradle.DefaultProjectIdentifier;
import org.gradle.tooling.provider.model.internal.BuildScopeModelBuilder;
import org.gradle.tooling.provider.model.internal.ToolingModelBuilderResultInternal;
import org.jspecify.annotations.NullMarked;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.ImmutableList.toImmutableList;
import static org.gradle.plugins.ide.internal.tooling.GradleBuildBuilder.GRADLE_BUILD_MODEL_NAME;
import static org.gradle.plugins.ide.internal.tooling.GradleBuildBuilder.addProjects;

@NullMarked
public class ResilientGradleBuildBuilder implements BuildScopeModelBuilder {
    private final BuildStateRegistry buildStateRegistry;
    private final BuildIncludeListener failedIncludedBuildsRegistry;
    private final FailureFactory failureFactory;

    public ResilientGradleBuildBuilder(
        BuildStateRegistry buildStateRegistry,
        BuildIncludeListener failedIncludedBuildsRegistry,
        FailureFactory failureFactory
    ) {
        this.buildStateRegistry = buildStateRegistry;
        this.failedIncludedBuildsRegistry = failedIncludedBuildsRegistry;
        this.failureFactory = failureFactory;
    }

    @Override
    public boolean canBuild(String modelName) {
        return GRADLE_BUILD_MODEL_NAME.equals(modelName);
    }

    @Override
    public ToolingModelBuilderResultInternal create(BuildState target) {
        return new ResilientGradleBuildCreator(target).create();
    }

    @NullMarked
    private class ResilientGradleBuildCreator {
        private final BuildState target;
        private final Map<BuildState, DefaultGradleBuild> all = new LinkedHashMap<>();
        private final Collection<Failure> failures = new LinkedHashSet<>();

        ResilientGradleBuildCreator(BuildState target) {
            this.target = target;
        }

        ToolingModelBuilderResultInternal create() {
            ensureProjectsLoaded(target);
            DefaultGradleBuild gradleBuild = convert(target);
            List<Failure> allFailures = failures.stream()
                .distinct()
                .collect(toImmutableList());
            return ToolingModelBuilderResultInternal.of(gradleBuild, allFailures);
        }

        protected void addIncludedBuilds(GradleInternal gradle, DefaultGradleBuild model) {
            for (IncludedBuildInternal reference : gradle.includedBuilds()) {
                BuildState target = reference.getTarget();
                if (target instanceof IncludedBuildState || target instanceof RootBuildState) {
                    model.addIncludedBuild(convert(target));
                } else {
                    throw new IllegalStateException("Unknown build type: " + reference.getClass().getName());
                }
            }
        }

        protected void addAllImportableBuilds(BuildState targetBuild, GradleInternal gradle, DefaultGradleBuild model) {
            if (gradle.getParent() == null) {
                List<DefaultGradleBuild> allBuilds = new ArrayList<>();
                buildStateRegistry.visitBuilds(buildState -> {
                    // Do not include the root build and only include builds that are intended to be imported into an IDE
                    if (buildState != targetBuild && buildState.isImportableBuild()) {
                        allBuilds.add(convert(buildState));
                    }
                });
                model.addBuilds(allBuilds);
            }
        }

        protected void ensureProjectsLoaded(BuildState target) {
            try {
                target.ensureProjectsLoaded();
            } catch (GradleException e) {
                failures.add(failureFactory.create(e));
            }
        }

        protected DefaultGradleBuild convert(BuildState targetBuild) {
            DefaultGradleBuild model = all.get(targetBuild);
            if (model != null) {
                return model;
            }
            model = new DefaultGradleBuild();
            all.put(targetBuild, model);

            ensureProjectsLoaded(targetBuild);

            Collection<SettingsInternal> brokenSettings = failedIncludedBuildsRegistry.getBrokenSettings();
            if (!brokenSettings.contains(targetBuild) && !brokenSettings.isEmpty()) {
                SettingsInternal settingsEntry = brokenSettings.iterator().next();
                ProjectDescriptor rootProject = settingsEntry.getRootProject();
                BasicGradleProject root = convertRoot(targetBuild, rootProject);
                model.setRootProject(root);
                model.addProject(root);
            }
            if (targetBuild instanceof IncludedBuildState) {
                model.setBuildIdentifier(new DefaultBuildIdentifier(((IncludedBuildState) targetBuild).getBuildDefinition().getBuildRootDir()));
            }

            GradleInternal gradle = targetBuild.getMutableModel();
            if (targetBuild.isProjectsLoaded()) {
                addProjects(targetBuild, model);
            }
            try {
                addFailedBuilds(targetBuild, model);
                addIncludedBuilds(gradle, model);
            } catch (IllegalStateException e) {
                //ignore, happens when included builds are not accessible, but we need this for resiliency
            }
            addAllImportableBuilds(targetBuild, gradle, model);
            return model;
        }

        protected BasicGradleProject convertRoot(BuildState owner, ProjectDescriptor project) {
            DefaultProjectIdentifier id = new DefaultProjectIdentifier(owner.getBuildRootDir(), project.getPath());
            return new BasicGradleProject()
                .setName(project.getName())
                .setProjectIdentifier(id)
                .setBuildTreePath(project.getPath())
                .setProjectDirectory(project.getProjectDir());
        }

        private void addFailedBuilds(BuildState targetBuild, DefaultGradleBuild model) {
            for (BuildState entry : failedIncludedBuildsRegistry.getBrokenBuilds()) {
                BuildState parent = entry.getParent();
                if (parent != null && parent.equals(targetBuild)) {
                    model.addIncludedBuild(convert(entry));
                }
            }
        }
    }
}
