/*
 * Copyright 2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.plugin.software.internal;

import org.gradle.api.Action;
import org.gradle.api.Project;
import org.gradle.api.internal.plugins.BuildModel;
import org.gradle.api.internal.plugins.DslBindingBuilder;
import org.gradle.api.internal.plugins.DslBindingBuilderInternal;
import org.gradle.api.internal.plugins.Definition;
import org.gradle.api.internal.plugins.ProjectFeatureApplicationContext;
import org.gradle.api.internal.plugins.ProjectFeatureBinding;
import org.gradle.api.internal.plugins.ProjectFeatureApplyAction;
import org.gradle.api.internal.plugins.ProjectTypeBindingBuilder;
import org.gradle.api.internal.plugins.ProjectTypeBindingBuilderInternal;
import org.gradle.api.internal.plugins.ProjectTypeApplyAction;
import org.gradle.api.internal.plugins.TargetTypeInformation;
import org.gradle.util.Path;

import java.util.ArrayList;
import java.util.List;

public class DefaultProjectTypeBindingBuilder implements ProjectTypeBindingBuilderInternal {
    private final List<DslBindingBuilderInternal<?, ?>> bindings = new ArrayList<>();

    @Override
    public <T extends Definition<V>, V extends BuildModel> DslBindingBuilder<T, V> bindProjectType(String name, Class<T> dslType, Class<V> buildModelType, ProjectTypeApplyAction<T, V> transform) {
        ProjectFeatureApplyAction<T, V, ?> featureTransform = (ProjectFeatureApplicationContext context, T definition, V buildModel, Object parentDefinition) ->
            transform.transform(context, definition, buildModel);

        DslBindingBuilderInternal<T, V> builder = new DefaultDslBindingBuilder<>(dslType,
            buildModelType,
            new TargetTypeInformation.DefinitionTargetTypeInformation<>(Project.class),
            Path.path(name),
            featureTransform
        );

        bindings.add(builder);
        return builder;
    }

    @Override
    public <T extends Definition<V>, V extends BuildModel> DslBindingBuilder<T, V> bindProjectType(String name, Class<T> dslType, ProjectTypeApplyAction<T, V> transform) {
        return bindProjectType(name, dslType, ModelTypeUtils.getBuildModelClass(dslType), transform);
    }

    public ProjectTypeBindingBuilder apply(Action<ProjectTypeBindingBuilder> configuration) {
        configuration.execute(this);
        return this;
    }

    @Override
    public List<ProjectFeatureBinding<?, ?>> build() {
        List<ProjectFeatureBinding<?, ?>> result = new ArrayList<>();
        for (DslBindingBuilderInternal<?, ?> binding : bindings) {
            result.add(binding.build());
        }
        return result;
    }
}
