/*
 * Copyright 2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.architecture.test;

import com.tngtech.archunit.base.DescribedPredicate;
import com.tngtech.archunit.core.domain.JavaClass;
import com.tngtech.archunit.junit.AnalyzeClasses;
import com.tngtech.archunit.junit.ArchTest;
import com.tngtech.archunit.lang.ArchRule;
import org.jspecify.annotations.NullMarked;

import static com.tngtech.archunit.base.DescribedPredicate.not;
import static com.tngtech.archunit.lang.syntax.ArchRuleDefinition.classes;
import static org.gradle.architecture.test.ArchUnitFixture.freeze;

@AnalyzeClasses(packages = "org.gradle")
@NullMarked
public class InternalClassesNamingValidationTest {
    @ArchTest
    public static final ArchRule internal_classes_should_use_internal_as_suffix_not_prefix =
        freeze(classes()
            .that().resideInAPackage("..internal..")
            .and().haveSimpleNameContaining("Internal")
            .and().resideOutsideOfPackage("org.gradle.tooling.internal.protocol..")
            .and(not(assignableToInterfaceInPackage("org.gradle.tooling.internal.protocol..")))
            .should().haveSimpleNameEndingWith("Internal")
            .andShould().haveSimpleNameNotStartingWith("Internal")
            .as("classes that are in 'internal' packages with 'Internal' in their name should use it as a suffix not as a prefix"));

    private static DescribedPredicate<JavaClass> assignableToInterfaceInPackage(String packagePattern) {
        return new DescribedPredicate<>("assignable to interface in " + packagePattern) {
            @Override
            public boolean test(JavaClass javaClass) {
                return javaClass.getAllRawInterfaces().stream()
                    .anyMatch(iface -> iface.getPackageName().matches(packagePattern.replace("..", ".*")));
            }
        };
    }
}
