/*
 * Copyright 2020 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.smoketests

import org.gradle.integtests.fixtures.executer.GradleContextualExecuter

import static org.gradle.testkit.runner.TaskOutcome.FROM_CACHE
import static org.gradle.testkit.runner.TaskOutcome.NO_SOURCE
import static org.gradle.testkit.runner.TaskOutcome.SKIPPED
import static org.gradle.testkit.runner.TaskOutcome.SUCCESS
import static org.gradle.testkit.runner.TaskOutcome.UP_TO_DATE

class AndroidSantaTrackerCachingSmokeTest extends AbstractAndroidSantaTrackerSmokeTest {

    def "can cache Santa Tracker Android application (agp=#agpVersion)"() {

        given:
        AGP_VERSIONS.assumeCurrentJavaVersionIsSupportedBy(agpVersion)

        and:
        def originalDir = temporaryFolder.createDir("original")
        def relocatedDir = temporaryFolder.createDir("relocated")

        and:
        setupCopyOfSantaTracker(originalDir)
        setupCopyOfSantaTracker(relocatedDir)

        when: 'clean build'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        def result = buildLocation(originalDir, agpVersion)

        then:
        if (GradleContextualExecuter.isConfigCache()) {
            result.assertConfigurationCacheStateStored()
        }

        when: 'up-to-date build, reusing configuration cache when enabled'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        result = buildCachedLocation(originalDir, agpVersion)

        then:
        if (GradleContextualExecuter.isConfigCache()) {
            result.assertConfigurationCacheStateLoaded()
        }

        when: 'up-to-date build, reusing configuration cache when enabled'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(originalDir, homeDir)
        result = buildCachedLocation(originalDir, agpVersion)

        then:
        if (GradleContextualExecuter.isConfigCache()) {
            result.assertConfigurationCacheStateLoaded()
        }

        when: 'clean cached build'
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(relocatedDir, homeDir)
        result = buildLocation(relocatedDir, agpVersion)

        then:
        if (GradleContextualExecuter.isConfigCache()) {
            result.assertConfigurationCacheStateStored()
        }

        and:
        def expectedResults = agpVersion.startsWith('8.4.') || agpVersion.startsWith('8.5.')
            ? AndroidPluginExpectationsOld8.EXPECTED_RESULTS_8_4
            : agpVersion.startsWith(('8.6.')) || agpVersion.startsWith(('8.7.')) || agpVersion.startsWith(('8.8.'))
            ? AndroidPluginExpectations8.EXPECTED_RESULTS_8_6
            : agpVersion.startsWith(('8.9.'))
            ? AndroidPluginExpectations8.EXPECTED_RESULTS_8_9
            : AndroidPluginExpectations8.EXPECTED_RESULTS_8_10

        verify(result, expectedResults)

        when: 'clean cached build, reusing configuration cache when enabled'
        runnerForLocation(relocatedDir, agpVersion, "clean").build()
        SantaTrackerConfigurationCacheWorkaround.beforeBuild(relocatedDir, homeDir)
        result = buildCachedLocation(relocatedDir, agpVersion)

        then:
        if (GradleContextualExecuter.isConfigCache()) {
            result.assertConfigurationCacheStateLoaded()
        }

        where:
        agpVersion << TestedVersions.androidGradle.versions
    }
}

class AndroidPluginExpectations8 {

    static final EXPECTED_RESULTS_8_10 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:copyDebugMergedManifest': SUCCESS,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': FROM_CACHE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': SUCCESS,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': SUCCESS,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': SUCCESS,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': SUCCESS,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': SUCCESS,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': FROM_CACHE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugAssets': SUCCESS,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': SUCCESS,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': SUCCESS,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': SUCCESS,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:copyDebugMergedManifest': SUCCESS,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': FROM_CACHE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': SUCCESS,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': SUCCESS,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': SUCCESS,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': SUCCESS,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': SUCCESS,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': FROM_CACHE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugAssets': SUCCESS,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': SUCCESS,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': SUCCESS,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:copyDebugMergedManifest': SUCCESS,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': FROM_CACHE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': SUCCESS,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': SUCCESS,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': SUCCESS,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:copyDebugMergedManifest': SUCCESS,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': FROM_CACHE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': SUCCESS,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': SUCCESS,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': SUCCESS,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': SUCCESS,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:copyDebugMergedManifest': SUCCESS,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': FROM_CACHE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': SUCCESS,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': SUCCESS,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': SUCCESS,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:copyDebugMergedManifest': SUCCESS,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': FROM_CACHE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': SUCCESS,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': SUCCESS,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': SUCCESS,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': SUCCESS,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': FROM_CACHE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugAssets': SUCCESS,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': SUCCESS,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': SUCCESS,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:copyDebugMergedManifest': SUCCESS,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': FROM_CACHE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': SUCCESS,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': SUCCESS,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': SUCCESS,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:copyDebugMergedManifest': SUCCESS,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': FROM_CACHE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': SUCCESS,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': SUCCESS,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': SUCCESS,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': SUCCESS,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': FROM_CACHE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': SUCCESS,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': SUCCESS,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': SUCCESS,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': SUCCESS,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:copyDebugMergedManifest': SUCCESS,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': FROM_CACHE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': SUCCESS,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': SUCCESS,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': SUCCESS,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': SUCCESS,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': FROM_CACHE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugAssets': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': SUCCESS,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': SUCCESS,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': SUCCESS,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': FROM_CACHE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': SUCCESS,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': SUCCESS,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': SUCCESS,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': SUCCESS,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]

    static final EXPECTED_RESULTS_8_9 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:copyDebugMergedManifest': SUCCESS,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': FROM_CACHE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': SUCCESS,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': SUCCESS,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': SUCCESS,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': SUCCESS,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': SUCCESS,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': FROM_CACHE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugAssets': SUCCESS,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': SUCCESS,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': SUCCESS,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': SUCCESS,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:copyDebugMergedManifest': SUCCESS,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': FROM_CACHE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': SUCCESS,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': SUCCESS,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': SUCCESS,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': SUCCESS,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': SUCCESS,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': FROM_CACHE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugAssets': SUCCESS,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': SUCCESS,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': SUCCESS,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:copyDebugMergedManifest': SUCCESS,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': FROM_CACHE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': SUCCESS,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': SUCCESS,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': SUCCESS,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:copyDebugMergedManifest': SUCCESS,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': FROM_CACHE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': SUCCESS,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': SUCCESS,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': SUCCESS,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': SUCCESS,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:copyDebugMergedManifest': SUCCESS,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': FROM_CACHE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': SUCCESS,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': SUCCESS,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': SUCCESS,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:copyDebugMergedManifest': SUCCESS,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': FROM_CACHE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': SUCCESS,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': SUCCESS,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': SUCCESS,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': SUCCESS,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': FROM_CACHE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugAssets': SUCCESS,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': SUCCESS,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': SUCCESS,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:copyDebugMergedManifest': SUCCESS,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': FROM_CACHE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': SUCCESS,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': SUCCESS,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': SUCCESS,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:copyDebugMergedManifest': SUCCESS,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': FROM_CACHE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': SUCCESS,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': SUCCESS,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': SUCCESS,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': SUCCESS,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': FROM_CACHE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': SUCCESS,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': SUCCESS,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': SUCCESS,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': SUCCESS,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:copyDebugMergedManifest': SUCCESS,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': FROM_CACHE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': SUCCESS,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': SUCCESS,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': SUCCESS,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': SUCCESS,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': FROM_CACHE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugAssets': SUCCESS,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': SUCCESS,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': SUCCESS,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': SUCCESS,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': FROM_CACHE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': SUCCESS,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': SUCCESS,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': SUCCESS,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': SUCCESS,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]

    static final EXPECTED_RESULTS_8_6 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:copyDebugMergedManifest': SUCCESS,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': FROM_CACHE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': SUCCESS,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': SUCCESS,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': SUCCESS,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': SUCCESS,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': SUCCESS,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': FROM_CACHE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': SUCCESS,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': SUCCESS,
        ':common:packageDebugAssets': SUCCESS,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': SUCCESS,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:copyDebugMergedManifest': SUCCESS,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': FROM_CACHE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': SUCCESS,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': SUCCESS,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': SUCCESS,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': SUCCESS,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': SUCCESS,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': FROM_CACHE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': SUCCESS,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': SUCCESS,
        ':doodles-lib:packageDebugAssets': SUCCESS,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:copyDebugMergedManifest': SUCCESS,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': FROM_CACHE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': SUCCESS,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': SUCCESS,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': SUCCESS,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:copyDebugMergedManifest': SUCCESS,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': FROM_CACHE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': SUCCESS,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': SUCCESS,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': SUCCESS,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': SUCCESS,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:copyDebugMergedManifest': SUCCESS,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': FROM_CACHE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': SUCCESS,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': SUCCESS,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': SUCCESS,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:copyDebugMergedManifest': SUCCESS,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': FROM_CACHE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': SUCCESS,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': SUCCESS,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': SUCCESS,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': SUCCESS,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': FROM_CACHE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': SUCCESS,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': SUCCESS,
        ':playgames:packageDebugAssets': SUCCESS,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:copyDebugMergedManifest': SUCCESS,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': FROM_CACHE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': SUCCESS,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': SUCCESS,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': SUCCESS,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:copyDebugMergedManifest': SUCCESS,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': FROM_CACHE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': SUCCESS,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': SUCCESS,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': SUCCESS,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': SUCCESS,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': FROM_CACHE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': SUCCESS,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': SUCCESS,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': SUCCESS,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': SUCCESS,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:copyDebugMergedManifest': SUCCESS,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': FROM_CACHE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': SUCCESS,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': SUCCESS,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': SUCCESS,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': SUCCESS,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': FROM_CACHE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': SUCCESS,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': SUCCESS,
        ':tracker:packageDebugAssets': SUCCESS,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': SUCCESS,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': FROM_CACHE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': SUCCESS,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': SUCCESS,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': SUCCESS,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': SUCCESS,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]
}

class AndroidPluginExpectationsOld8 {

    static final EXPECTED_RESULTS_8_4 = [
        ':cityquiz:assembleDebug': SUCCESS,
        ':cityquiz:checkDebugAarMetadata': SUCCESS,
        ':cityquiz:checkDebugDuplicateClasses': SUCCESS,
        ':cityquiz:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':cityquiz:compileDebugJavaWithJavac': FROM_CACHE,
        ':cityquiz:compileDebugKotlin': FROM_CACHE,
        ':cityquiz:compileDebugShaders': NO_SOURCE,
        ':cityquiz:compressDebugAssets': FROM_CACHE,
        ':cityquiz:copyDebugMergedManifest': SUCCESS,
        ':cityquiz:createDebugApkListingFileRedirect': SUCCESS,
        ':cityquiz:desugarDebugFileDependencies': FROM_CACHE,
        ':cityquiz:dexBuilderDebug': FROM_CACHE,
        ':cityquiz:extractDeepLinksDebug': FROM_CACHE,
        ':cityquiz:featureDebugWriter': SUCCESS,
        ':cityquiz:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':cityquiz:generateDebugAssets': UP_TO_DATE,
        ':cityquiz:generateDebugBuildConfig': FROM_CACHE,
        ':cityquiz:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':cityquiz:generateDebugResValues': FROM_CACHE,
        ':cityquiz:generateDebugResources': FROM_CACHE,
        ':cityquiz:javaPreCompileDebug': FROM_CACHE,
        ':cityquiz:mapDebugSourceSetPaths': SUCCESS,
        ':cityquiz:mergeDebugAssets': SUCCESS,
        ':cityquiz:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':cityquiz:mergeDebugJavaResource': SUCCESS,
        ':cityquiz:mergeDebugJniLibFolders': SUCCESS,
        ':cityquiz:mergeDebugNativeLibs': NO_SOURCE,
        ':cityquiz:mergeDebugResources': FROM_CACHE,
        ':cityquiz:mergeDebugShaders': SUCCESS,
        ':cityquiz:mergeExtDexDebug': FROM_CACHE,
        ':cityquiz:mergeLibDexDebug': FROM_CACHE,
        ':cityquiz:mergeProjectDexDebug': FROM_CACHE,
        ':cityquiz:packageDebug': SUCCESS,
        ':cityquiz:preBuild': UP_TO_DATE,
        ':cityquiz:preDebugBuild': UP_TO_DATE,
        ':cityquiz:processApplicationManifestDebugForBundle': SUCCESS,
        ':cityquiz:processDebugJavaRes': SUCCESS,
        ':cityquiz:processDebugMainManifest': FROM_CACHE,
        ':cityquiz:processDebugManifestForPackage': FROM_CACHE,
        ':cityquiz:processDebugResources': FROM_CACHE,
        ':cityquiz:processManifestDebugForFeature': FROM_CACHE,
        ':cityquiz:stripDebugDebugSymbols': NO_SOURCE,
        ':common:assembleDebug': SUCCESS,
        ':common:bundleDebugAar': SUCCESS,
        ':common:bundleLibCompileToJarDebug': SUCCESS,
        ':common:bundleLibRuntimeToJarDebug': SUCCESS,
        ':common:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':common:compileDebugJavaWithJavac': FROM_CACHE,
        ':common:compileDebugKotlin': FROM_CACHE,
        ':common:compileDebugLibraryResources': FROM_CACHE,
        ':common:compileDebugShaders': NO_SOURCE,
        ':common:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':common:copyDebugJniLibsProjectOnly': SUCCESS,
        ':common:createFullJarDebug': SUCCESS,
        ':common:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':common:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':common:extractDebugAnnotations': FROM_CACHE,
        ':common:extractDeepLinksDebug': FROM_CACHE,
        ':common:extractDeepLinksForAarDebug': FROM_CACHE,
        ':common:generateDebugAssets': UP_TO_DATE,
        ':common:generateDebugBuildConfig': FROM_CACHE,
        ':common:generateDebugRFile': FROM_CACHE,
        ':common:generateDebugResValues': FROM_CACHE,
        ':common:generateDebugResources': FROM_CACHE,
        ':common:javaPreCompileDebug': FROM_CACHE,
        ':common:mergeDebugConsumerProguardFiles': SUCCESS,
        ':common:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':common:mergeDebugJavaResource': SUCCESS,
        ':common:mergeDebugJniLibFolders': SUCCESS,
        ':common:mergeDebugNativeLibs': NO_SOURCE,
        ':common:mergeDebugShaders': SUCCESS,
        ':common:packageDebugAssets': SUCCESS,
        ':common:packageDebugResources': FROM_CACHE,
        ':common:parseDebugLocalResources': FROM_CACHE,
        ':common:preBuild': UP_TO_DATE,
        ':common:preDebugBuild': UP_TO_DATE,
        ':common:prepareDebugArtProfile': SUCCESS,
        ':common:prepareLintJarForPublish': SUCCESS,
        ':common:processDebugJavaRes': SUCCESS,
        ':common:processDebugManifest': FROM_CACHE,
        ':common:stripDebugDebugSymbols': NO_SOURCE,
        ':common:syncDebugLibJars': FROM_CACHE,
        ':common:writeDebugAarMetadata': SUCCESS,
        ':dasherdancer:assembleDebug': SUCCESS,
        ':dasherdancer:checkDebugAarMetadata': SUCCESS,
        ':dasherdancer:checkDebugDuplicateClasses': SUCCESS,
        ':dasherdancer:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':dasherdancer:compileDebugJavaWithJavac': NO_SOURCE,
        ':dasherdancer:compileDebugKotlin': FROM_CACHE,
        ':dasherdancer:compileDebugShaders': NO_SOURCE,
        ':dasherdancer:compressDebugAssets': FROM_CACHE,
        ':dasherdancer:copyDebugMergedManifest': SUCCESS,
        ':dasherdancer:createDebugApkListingFileRedirect': SUCCESS,
        ':dasherdancer:desugarDebugFileDependencies': FROM_CACHE,
        ':dasherdancer:dexBuilderDebug': FROM_CACHE,
        ':dasherdancer:extractDeepLinksDebug': FROM_CACHE,
        ':dasherdancer:featureDebugWriter': SUCCESS,
        ':dasherdancer:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':dasherdancer:generateDebugAssets': UP_TO_DATE,
        ':dasherdancer:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':dasherdancer:generateDebugResValues': FROM_CACHE,
        ':dasherdancer:generateDebugResources': FROM_CACHE,
        ':dasherdancer:javaPreCompileDebug': FROM_CACHE,
        ':dasherdancer:mapDebugSourceSetPaths': SUCCESS,
        ':dasherdancer:mergeDebugAssets': SUCCESS,
        ':dasherdancer:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':dasherdancer:mergeDebugJavaResource': SUCCESS,
        ':dasherdancer:mergeDebugJniLibFolders': SUCCESS,
        ':dasherdancer:mergeDebugNativeLibs': NO_SOURCE,
        ':dasherdancer:mergeDebugResources': FROM_CACHE,
        ':dasherdancer:mergeDebugShaders': SUCCESS,
        ':dasherdancer:mergeExtDexDebug': FROM_CACHE,
        ':dasherdancer:mergeLibDexDebug': FROM_CACHE,
        ':dasherdancer:mergeProjectDexDebug': FROM_CACHE,
        ':dasherdancer:packageDebug': SUCCESS,
        ':dasherdancer:preBuild': UP_TO_DATE,
        ':dasherdancer:preDebugBuild': UP_TO_DATE,
        ':dasherdancer:processApplicationManifestDebugForBundle': SUCCESS,
        ':dasherdancer:processDebugJavaRes': SUCCESS,
        ':dasherdancer:processDebugMainManifest': FROM_CACHE,
        ':dasherdancer:processDebugManifestForPackage': FROM_CACHE,
        ':dasherdancer:processDebugResources': FROM_CACHE,
        ':dasherdancer:processManifestDebugForFeature': FROM_CACHE,
        ':dasherdancer:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:assembleDebug': SUCCESS,
        ':doodles-lib:bundleDebugAar': SUCCESS,
        ':doodles-lib:bundleLibCompileToJarDebug': SUCCESS,
        ':doodles-lib:bundleLibRuntimeToJarDebug': SUCCESS,
        ':doodles-lib:compileDebugJavaWithJavac': FROM_CACHE,
        ':doodles-lib:compileDebugLibraryResources': FROM_CACHE,
        ':doodles-lib:compileDebugShaders': NO_SOURCE,
        ':doodles-lib:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':doodles-lib:copyDebugJniLibsProjectOnly': SUCCESS,
        ':doodles-lib:createFullJarDebug': SUCCESS,
        ':doodles-lib:extractDebugAnnotations': FROM_CACHE,
        ':doodles-lib:extractDeepLinksDebug': FROM_CACHE,
        ':doodles-lib:extractDeepLinksForAarDebug': FROM_CACHE,
        ':doodles-lib:generateDebugAssets': UP_TO_DATE,
        ':doodles-lib:generateDebugRFile': FROM_CACHE,
        ':doodles-lib:generateDebugResValues': FROM_CACHE,
        ':doodles-lib:generateDebugResources': FROM_CACHE,
        ':doodles-lib:javaPreCompileDebug': FROM_CACHE,
        ':doodles-lib:mergeDebugConsumerProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':doodles-lib:mergeDebugJavaResource': SUCCESS,
        ':doodles-lib:mergeDebugJniLibFolders': SUCCESS,
        ':doodles-lib:mergeDebugNativeLibs': NO_SOURCE,
        ':doodles-lib:mergeDebugShaders': SUCCESS,
        ':doodles-lib:packageDebugAssets': SUCCESS,
        ':doodles-lib:packageDebugResources': FROM_CACHE,
        ':doodles-lib:parseDebugLocalResources': FROM_CACHE,
        ':doodles-lib:preBuild': UP_TO_DATE,
        ':doodles-lib:preDebugBuild': UP_TO_DATE,
        ':doodles-lib:prepareDebugArtProfile': SUCCESS,
        ':doodles-lib:prepareLintJarForPublish': SUCCESS,
        ':doodles-lib:processDebugJavaRes': NO_SOURCE,
        ':doodles-lib:processDebugManifest': FROM_CACHE,
        ':doodles-lib:stripDebugDebugSymbols': NO_SOURCE,
        ':doodles-lib:syncDebugLibJars': FROM_CACHE,
        ':doodles-lib:writeDebugAarMetadata': SUCCESS,
        ':gumball:assembleDebug': SUCCESS,
        ':gumball:checkDebugAarMetadata': SUCCESS,
        ':gumball:checkDebugDuplicateClasses': SUCCESS,
        ':gumball:compileDebugJavaWithJavac': FROM_CACHE,
        ':gumball:compileDebugShaders': NO_SOURCE,
        ':gumball:compressDebugAssets': FROM_CACHE,
        ':gumball:copyDebugMergedManifest': SUCCESS,
        ':gumball:createDebugApkListingFileRedirect': SUCCESS,
        ':gumball:desugarDebugFileDependencies': FROM_CACHE,
        ':gumball:dexBuilderDebug': FROM_CACHE,
        ':gumball:extractDeepLinksDebug': FROM_CACHE,
        ':gumball:featureDebugWriter': SUCCESS,
        ':gumball:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':gumball:generateDebugAssets': UP_TO_DATE,
        ':gumball:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':gumball:generateDebugResValues': FROM_CACHE,
        ':gumball:generateDebugResources': FROM_CACHE,
        ':gumball:javaPreCompileDebug': FROM_CACHE,
        ':gumball:mapDebugSourceSetPaths': SUCCESS,
        ':gumball:mergeDebugAssets': SUCCESS,
        ':gumball:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':gumball:mergeDebugJavaResource': SUCCESS,
        ':gumball:mergeDebugJniLibFolders': SUCCESS,
        ':gumball:mergeDebugNativeLibs': NO_SOURCE,
        ':gumball:mergeDebugResources': FROM_CACHE,
        ':gumball:mergeDebugShaders': SUCCESS,
        ':gumball:mergeExtDexDebug': FROM_CACHE,
        ':gumball:mergeLibDexDebug': FROM_CACHE,
        ':gumball:mergeProjectDexDebug': FROM_CACHE,
        ':gumball:packageDebug': SUCCESS,
        ':gumball:preBuild': UP_TO_DATE,
        ':gumball:preDebugBuild': UP_TO_DATE,
        ':gumball:processApplicationManifestDebugForBundle': SUCCESS,
        ':gumball:processDebugJavaRes': NO_SOURCE,
        ':gumball:processDebugMainManifest': FROM_CACHE,
        ':gumball:processDebugManifestForPackage': FROM_CACHE,
        ':gumball:processDebugResources': FROM_CACHE,
        ':gumball:processManifestDebugForFeature': FROM_CACHE,
        ':gumball:stripDebugDebugSymbols': NO_SOURCE,
        ':jetpack:assembleDebug': SUCCESS,
        ':jetpack:checkDebugAarMetadata': SUCCESS,
        ':jetpack:checkDebugDuplicateClasses': SUCCESS,
        ':jetpack:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':jetpack:compileDebugJavaWithJavac': NO_SOURCE,
        ':jetpack:compileDebugKotlin': FROM_CACHE,
        ':jetpack:compileDebugShaders': NO_SOURCE,
        ':jetpack:compressDebugAssets': FROM_CACHE,
        ':jetpack:copyDebugMergedManifest': SUCCESS,
        ':jetpack:createDebugApkListingFileRedirect': SUCCESS,
        ':jetpack:desugarDebugFileDependencies': FROM_CACHE,
        ':jetpack:dexBuilderDebug': FROM_CACHE,
        ':jetpack:extractDeepLinksDebug': FROM_CACHE,
        ':jetpack:featureDebugWriter': SUCCESS,
        ':jetpack:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':jetpack:generateDebugAssets': UP_TO_DATE,
        ':jetpack:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':jetpack:generateDebugResValues': FROM_CACHE,
        ':jetpack:generateDebugResources': FROM_CACHE,
        ':jetpack:javaPreCompileDebug': FROM_CACHE,
        ':jetpack:mapDebugSourceSetPaths': SUCCESS,
        ':jetpack:mergeDebugAssets': SUCCESS,
        ':jetpack:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':jetpack:mergeDebugJavaResource': SUCCESS,
        ':jetpack:mergeDebugJniLibFolders': SUCCESS,
        ':jetpack:mergeDebugNativeLibs': NO_SOURCE,
        ':jetpack:mergeDebugResources': FROM_CACHE,
        ':jetpack:mergeDebugShaders': SUCCESS,
        ':jetpack:mergeExtDexDebug': FROM_CACHE,
        ':jetpack:mergeLibDexDebug': FROM_CACHE,
        ':jetpack:mergeProjectDexDebug': FROM_CACHE,
        ':jetpack:packageDebug': SUCCESS,
        ':jetpack:preBuild': UP_TO_DATE,
        ':jetpack:preDebugBuild': UP_TO_DATE,
        ':jetpack:processApplicationManifestDebugForBundle': SUCCESS,
        ':jetpack:processDebugJavaRes': SUCCESS,
        ':jetpack:processDebugMainManifest': FROM_CACHE,
        ':jetpack:processDebugManifestForPackage': FROM_CACHE,
        ':jetpack:processDebugResources': FROM_CACHE,
        ':jetpack:processManifestDebugForFeature': FROM_CACHE,
        ':jetpack:stripDebugDebugSymbols': NO_SOURCE,
        ':memory:assembleDebug': SUCCESS,
        ':memory:checkDebugAarMetadata': SUCCESS,
        ':memory:checkDebugDuplicateClasses': SUCCESS,
        ':memory:compileDebugJavaWithJavac': FROM_CACHE,
        ':memory:compileDebugShaders': NO_SOURCE,
        ':memory:compressDebugAssets': FROM_CACHE,
        ':memory:copyDebugMergedManifest': SUCCESS,
        ':memory:createDebugApkListingFileRedirect': SUCCESS,
        ':memory:desugarDebugFileDependencies': FROM_CACHE,
        ':memory:dexBuilderDebug': FROM_CACHE,
        ':memory:extractDeepLinksDebug': FROM_CACHE,
        ':memory:featureDebugWriter': SUCCESS,
        ':memory:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':memory:generateDebugAssets': UP_TO_DATE,
        ':memory:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':memory:generateDebugResValues': FROM_CACHE,
        ':memory:generateDebugResources': FROM_CACHE,
        ':memory:javaPreCompileDebug': FROM_CACHE,
        ':memory:mapDebugSourceSetPaths': SUCCESS,
        ':memory:mergeDebugAssets': SUCCESS,
        ':memory:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':memory:mergeDebugJavaResource': SUCCESS,
        ':memory:mergeDebugJniLibFolders': SUCCESS,
        ':memory:mergeDebugNativeLibs': NO_SOURCE,
        ':memory:mergeDebugResources': FROM_CACHE,
        ':memory:mergeDebugShaders': SUCCESS,
        ':memory:mergeExtDexDebug': FROM_CACHE,
        ':memory:mergeLibDexDebug': FROM_CACHE,
        ':memory:mergeProjectDexDebug': FROM_CACHE,
        ':memory:packageDebug': SUCCESS,
        ':memory:preBuild': UP_TO_DATE,
        ':memory:preDebugBuild': UP_TO_DATE,
        ':memory:processApplicationManifestDebugForBundle': SUCCESS,
        ':memory:processDebugJavaRes': NO_SOURCE,
        ':memory:processDebugMainManifest': FROM_CACHE,
        ':memory:processDebugManifestForPackage': FROM_CACHE,
        ':memory:processDebugResources': FROM_CACHE,
        ':memory:processManifestDebugForFeature': FROM_CACHE,
        ':memory:stripDebugDebugSymbols': NO_SOURCE,
        ':penguinswim:assembleDebug': SUCCESS,
        ':penguinswim:checkDebugAarMetadata': SUCCESS,
        ':penguinswim:checkDebugDuplicateClasses': SUCCESS,
        ':penguinswim:compileDebugJavaWithJavac': FROM_CACHE,
        ':penguinswim:compileDebugShaders': NO_SOURCE,
        ':penguinswim:compressDebugAssets': FROM_CACHE,
        ':penguinswim:copyDebugMergedManifest': SUCCESS,
        ':penguinswim:createDebugApkListingFileRedirect': SUCCESS,
        ':penguinswim:desugarDebugFileDependencies': FROM_CACHE,
        ':penguinswim:dexBuilderDebug': FROM_CACHE,
        ':penguinswim:extractDeepLinksDebug': FROM_CACHE,
        ':penguinswim:featureDebugWriter': SUCCESS,
        ':penguinswim:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':penguinswim:generateDebugAssets': UP_TO_DATE,
        ':penguinswim:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':penguinswim:generateDebugResValues': FROM_CACHE,
        ':penguinswim:generateDebugResources': FROM_CACHE,
        ':penguinswim:javaPreCompileDebug': FROM_CACHE,
        ':penguinswim:mapDebugSourceSetPaths': SUCCESS,
        ':penguinswim:mergeDebugAssets': SUCCESS,
        ':penguinswim:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':penguinswim:mergeDebugJavaResource': SUCCESS,
        ':penguinswim:mergeDebugJniLibFolders': SUCCESS,
        ':penguinswim:mergeDebugNativeLibs': NO_SOURCE,
        ':penguinswim:mergeDebugResources': FROM_CACHE,
        ':penguinswim:mergeDebugShaders': SUCCESS,
        ':penguinswim:mergeExtDexDebug': FROM_CACHE,
        ':penguinswim:mergeLibDexDebug': FROM_CACHE,
        ':penguinswim:mergeProjectDexDebug': FROM_CACHE,
        ':penguinswim:packageDebug': SUCCESS,
        ':penguinswim:preBuild': UP_TO_DATE,
        ':penguinswim:preDebugBuild': UP_TO_DATE,
        ':penguinswim:processApplicationManifestDebugForBundle': SUCCESS,
        ':penguinswim:processDebugJavaRes': NO_SOURCE,
        ':penguinswim:processDebugMainManifest': FROM_CACHE,
        ':penguinswim:processDebugManifestForPackage': FROM_CACHE,
        ':penguinswim:processDebugResources': FROM_CACHE,
        ':penguinswim:processManifestDebugForFeature': FROM_CACHE,
        ':penguinswim:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:assembleDebug': SUCCESS,
        ':playgames:bundleDebugAar': SUCCESS,
        ':playgames:bundleLibCompileToJarDebug': SUCCESS,
        ':playgames:bundleLibRuntimeToJarDebug': SUCCESS,
        ':playgames:compileDebugJavaWithJavac': FROM_CACHE,
        ':playgames:compileDebugLibraryResources': FROM_CACHE,
        ':playgames:compileDebugShaders': NO_SOURCE,
        ':playgames:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':playgames:copyDebugJniLibsProjectOnly': SUCCESS,
        ':playgames:createFullJarDebug': SUCCESS,
        ':playgames:extractDebugAnnotations': FROM_CACHE,
        ':playgames:extractDeepLinksDebug': FROM_CACHE,
        ':playgames:extractDeepLinksForAarDebug': FROM_CACHE,
        ':playgames:generateDebugAssets': UP_TO_DATE,
        ':playgames:generateDebugRFile': FROM_CACHE,
        ':playgames:generateDebugResValues': FROM_CACHE,
        ':playgames:generateDebugResources': FROM_CACHE,
        ':playgames:javaPreCompileDebug': FROM_CACHE,
        ':playgames:mergeDebugConsumerProguardFiles': SUCCESS,
        ':playgames:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':playgames:mergeDebugJavaResource': SUCCESS,
        ':playgames:mergeDebugJniLibFolders': SUCCESS,
        ':playgames:mergeDebugNativeLibs': NO_SOURCE,
        ':playgames:mergeDebugShaders': SUCCESS,
        ':playgames:packageDebugAssets': SUCCESS,
        ':playgames:packageDebugResources': FROM_CACHE,
        ':playgames:parseDebugLocalResources': FROM_CACHE,
        ':playgames:preBuild': UP_TO_DATE,
        ':playgames:preDebugBuild': UP_TO_DATE,
        ':playgames:prepareDebugArtProfile': SUCCESS,
        ':playgames:prepareLintJarForPublish': SUCCESS,
        ':playgames:processDebugJavaRes': NO_SOURCE,
        ':playgames:processDebugManifest': FROM_CACHE,
        ':playgames:stripDebugDebugSymbols': NO_SOURCE,
        ':playgames:syncDebugLibJars': FROM_CACHE,
        ':playgames:writeDebugAarMetadata': SUCCESS,
        ':presenttoss:assembleDebug': SUCCESS,
        ':presenttoss:checkDebugAarMetadata': SUCCESS,
        ':presenttoss:checkDebugDuplicateClasses': SUCCESS,
        ':presenttoss:compileDebugJavaWithJavac': FROM_CACHE,
        ':presenttoss:compileDebugShaders': NO_SOURCE,
        ':presenttoss:compressDebugAssets': FROM_CACHE,
        ':presenttoss:copyDebugMergedManifest': SUCCESS,
        ':presenttoss:createDebugApkListingFileRedirect': SUCCESS,
        ':presenttoss:desugarDebugFileDependencies': FROM_CACHE,
        ':presenttoss:dexBuilderDebug': FROM_CACHE,
        ':presenttoss:extractDeepLinksDebug': FROM_CACHE,
        ':presenttoss:featureDebugWriter': SUCCESS,
        ':presenttoss:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':presenttoss:generateDebugAssets': UP_TO_DATE,
        ':presenttoss:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':presenttoss:generateDebugResValues': FROM_CACHE,
        ':presenttoss:generateDebugResources': FROM_CACHE,
        ':presenttoss:javaPreCompileDebug': FROM_CACHE,
        ':presenttoss:mapDebugSourceSetPaths': SUCCESS,
        ':presenttoss:mergeDebugAssets': SUCCESS,
        ':presenttoss:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':presenttoss:mergeDebugJavaResource': SUCCESS,
        ':presenttoss:mergeDebugJniLibFolders': SUCCESS,
        ':presenttoss:mergeDebugNativeLibs': NO_SOURCE,
        ':presenttoss:mergeDebugResources': FROM_CACHE,
        ':presenttoss:mergeDebugShaders': SUCCESS,
        ':presenttoss:mergeExtDexDebug': FROM_CACHE,
        ':presenttoss:mergeLibDexDebug': FROM_CACHE,
        ':presenttoss:mergeProjectDexDebug': FROM_CACHE,
        ':presenttoss:packageDebug': SUCCESS,
        ':presenttoss:preBuild': UP_TO_DATE,
        ':presenttoss:preDebugBuild': UP_TO_DATE,
        ':presenttoss:processApplicationManifestDebugForBundle': SUCCESS,
        ':presenttoss:processDebugJavaRes': NO_SOURCE,
        ':presenttoss:processDebugMainManifest': FROM_CACHE,
        ':presenttoss:processDebugManifestForPackage': FROM_CACHE,
        ':presenttoss:processDebugResources': FROM_CACHE,
        ':presenttoss:processManifestDebugForFeature': FROM_CACHE,
        ':presenttoss:stripDebugDebugSymbols': NO_SOURCE,
        ':rocketsleigh:assembleDebug': SUCCESS,
        ':rocketsleigh:checkDebugAarMetadata': SUCCESS,
        ':rocketsleigh:checkDebugDuplicateClasses': SUCCESS,
        ':rocketsleigh:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':rocketsleigh:compileDebugJavaWithJavac': FROM_CACHE,
        ':rocketsleigh:compileDebugKotlin': FROM_CACHE,
        ':rocketsleigh:compileDebugShaders': NO_SOURCE,
        ':rocketsleigh:compressDebugAssets': FROM_CACHE,
        ':rocketsleigh:copyDebugMergedManifest': SUCCESS,
        ':rocketsleigh:createDebugApkListingFileRedirect': SUCCESS,
        ':rocketsleigh:desugarDebugFileDependencies': FROM_CACHE,
        ':rocketsleigh:dexBuilderDebug': FROM_CACHE,
        ':rocketsleigh:extractDeepLinksDebug': FROM_CACHE,
        ':rocketsleigh:featureDebugWriter': SUCCESS,
        ':rocketsleigh:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':rocketsleigh:generateDebugAssets': UP_TO_DATE,
        ':rocketsleigh:generateDebugBuildConfig': FROM_CACHE,
        ':rocketsleigh:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':rocketsleigh:generateDebugResValues': FROM_CACHE,
        ':rocketsleigh:generateDebugResources': FROM_CACHE,
        ':rocketsleigh:javaPreCompileDebug': FROM_CACHE,
        ':rocketsleigh:mapDebugSourceSetPaths': SUCCESS,
        ':rocketsleigh:mergeDebugAssets': SUCCESS,
        ':rocketsleigh:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':rocketsleigh:mergeDebugJavaResource': SUCCESS,
        ':rocketsleigh:mergeDebugJniLibFolders': SUCCESS,
        ':rocketsleigh:mergeDebugNativeLibs': NO_SOURCE,
        ':rocketsleigh:mergeDebugResources': FROM_CACHE,
        ':rocketsleigh:mergeDebugShaders': SUCCESS,
        ':rocketsleigh:mergeExtDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeLibDexDebug': FROM_CACHE,
        ':rocketsleigh:mergeProjectDexDebug': FROM_CACHE,
        ':rocketsleigh:packageDebug': SUCCESS,
        ':rocketsleigh:preBuild': UP_TO_DATE,
        ':rocketsleigh:preDebugBuild': UP_TO_DATE,
        ':rocketsleigh:processApplicationManifestDebugForBundle': SUCCESS,
        ':rocketsleigh:processDebugJavaRes': SUCCESS,
        ':rocketsleigh:processDebugMainManifest': FROM_CACHE,
        ':rocketsleigh:processDebugManifestForPackage': FROM_CACHE,
        ':rocketsleigh:processDebugResources': FROM_CACHE,
        ':rocketsleigh:processManifestDebugForFeature': FROM_CACHE,
        ':rocketsleigh:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:assembleDebug': SUCCESS,
        ':santa-tracker:bundleDebugClassesToCompileJar': SUCCESS,
        ':santa-tracker:bundleDebugClassesToRuntimeJar': SUCCESS,
        ':santa-tracker:checkDebugAarMetadata': SUCCESS,
        ':santa-tracker:checkDebugDuplicateClasses': SUCCESS,
        ':santa-tracker:checkDebugLibraries': SUCCESS,
        ':santa-tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':santa-tracker:compileDebugJavaWithJavac': FROM_CACHE,
        ':santa-tracker:compileDebugKotlin': FROM_CACHE,
        ':santa-tracker:compileDebugShaders': NO_SOURCE,
        ':santa-tracker:compressDebugAssets': FROM_CACHE,
        ':santa-tracker:createDebugApkListingFileRedirect': SUCCESS,
        ':santa-tracker:createDebugCompatibleScreenManifests': SUCCESS,
        ':santa-tracker:dataBindingGenBaseClassesDebug': FROM_CACHE,
        ':santa-tracker:dataBindingMergeDependencyArtifactsDebug': SUCCESS,
        ':santa-tracker:desugarDebugFileDependencies': FROM_CACHE,
        ':santa-tracker:dexBuilderDebug': FROM_CACHE,
        ':santa-tracker:extractDeepLinksDebug': FROM_CACHE,
        ':santa-tracker:generateDebugAssets': UP_TO_DATE,
        ':santa-tracker:generateDebugBuildConfig': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureMetadata': FROM_CACHE,
        ':santa-tracker:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':santa-tracker:generateDebugResValues': FROM_CACHE,
        ':santa-tracker:generateDebugResources': FROM_CACHE,
        ':santa-tracker:handleDebugMicroApk': FROM_CACHE,
        ':santa-tracker:javaPreCompileDebug': FROM_CACHE,
        ':santa-tracker:kaptDebugKotlin': FROM_CACHE,
        ':santa-tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':santa-tracker:mapDebugSourceSetPaths': SUCCESS,
        ':santa-tracker:mergeDebugAssets': SUCCESS,
        ':santa-tracker:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':santa-tracker:mergeDebugJavaResource': SUCCESS,
        ':santa-tracker:mergeDebugJniLibFolders': SUCCESS,
        ':santa-tracker:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':santa-tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':santa-tracker:mergeDebugResources': FROM_CACHE,
        ':santa-tracker:mergeDebugShaders': SUCCESS,
        ':santa-tracker:mergeDebugStartupProfile': SUCCESS,
        ':santa-tracker:mergeExtDexDebug': FROM_CACHE,
        ':santa-tracker:mergeLibDexDebug': FROM_CACHE,
        ':santa-tracker:mergeProjectDexDebug': FROM_CACHE,
        ':santa-tracker:packageDebug': SUCCESS,
        ':santa-tracker:preBuild': UP_TO_DATE,
        ':santa-tracker:preDebugBuild': SUCCESS,
        ':santa-tracker:processDebugJavaRes': SUCCESS,
        ':santa-tracker:processDebugMainManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifest': FROM_CACHE,
        ':santa-tracker:processDebugManifestForPackage': FROM_CACHE,
        ':santa-tracker:processDebugResources': FROM_CACHE,
        ':santa-tracker:signingConfigWriterDebug': SUCCESS,
        ':santa-tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':santa-tracker:validateSigningDebug': SUCCESS,
        ':santa-tracker:writeDebugAppMetadata': SUCCESS,
        ':santa-tracker:writeDebugModuleMetadata': SUCCESS,
        ':santa-tracker:writeDebugSigningConfigVersions': SUCCESS,
        ':snowballrun:assembleDebug': SUCCESS,
        ':snowballrun:checkDebugAarMetadata': SUCCESS,
        ':snowballrun:checkDebugDuplicateClasses': SUCCESS,
        ':snowballrun:compileDebugJavaWithJavac': FROM_CACHE,
        ':snowballrun:compileDebugShaders': NO_SOURCE,
        ':snowballrun:compressDebugAssets': FROM_CACHE,
        ':snowballrun:copyDebugMergedManifest': SUCCESS,
        ':snowballrun:createDebugApkListingFileRedirect': SUCCESS,
        ':snowballrun:desugarDebugFileDependencies': FROM_CACHE,
        ':snowballrun:dexBuilderDebug': FROM_CACHE,
        ':snowballrun:extractDeepLinksDebug': FROM_CACHE,
        ':snowballrun:featureDebugWriter': SUCCESS,
        ':snowballrun:featureGlobalSyntheticsDebugMerge': SUCCESS,
        ':snowballrun:generateDebugAssets': UP_TO_DATE,
        ':snowballrun:generateDebugFeatureTransitiveDeps': SUCCESS,
        ':snowballrun:generateDebugResValues': FROM_CACHE,
        ':snowballrun:generateDebugResources': FROM_CACHE,
        ':snowballrun:javaPreCompileDebug': FROM_CACHE,
        ':snowballrun:mapDebugSourceSetPaths': SUCCESS,
        ':snowballrun:mergeDebugAssets': SUCCESS,
        ':snowballrun:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':snowballrun:mergeDebugJavaResource': SUCCESS,
        ':snowballrun:mergeDebugJniLibFolders': SUCCESS,
        ':snowballrun:mergeDebugNativeLibs': NO_SOURCE,
        ':snowballrun:mergeDebugResources': FROM_CACHE,
        ':snowballrun:mergeDebugShaders': SUCCESS,
        ':snowballrun:mergeExtDexDebug': FROM_CACHE,
        ':snowballrun:mergeLibDexDebug': FROM_CACHE,
        ':snowballrun:mergeProjectDexDebug': FROM_CACHE,
        ':snowballrun:packageDebug': SUCCESS,
        ':snowballrun:preBuild': UP_TO_DATE,
        ':snowballrun:preDebugBuild': UP_TO_DATE,
        ':snowballrun:processApplicationManifestDebugForBundle': SUCCESS,
        ':snowballrun:processDebugJavaRes': NO_SOURCE,
        ':snowballrun:processDebugMainManifest': FROM_CACHE,
        ':snowballrun:processDebugManifestForPackage': FROM_CACHE,
        ':snowballrun:processDebugResources': FROM_CACHE,
        ':snowballrun:processManifestDebugForFeature': FROM_CACHE,
        ':snowballrun:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:assembleDebug': SUCCESS,
        ':tracker:bundleDebugAar': SUCCESS,
        ':tracker:bundleLibCompileToJarDebug': SUCCESS,
        ':tracker:bundleLibRuntimeToJarDebug': SUCCESS,
        ':tracker:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':tracker:compileDebugJavaWithJavac': SUCCESS,
        ':tracker:compileDebugKotlin': FROM_CACHE,
        ':tracker:compileDebugLibraryResources': FROM_CACHE,
        ':tracker:compileDebugShaders': NO_SOURCE,
        ':tracker:copyDebugJniLibsProjectAndLocalJars': SUCCESS,
        ':tracker:copyDebugJniLibsProjectOnly': SUCCESS,
        ':tracker:createFullJarDebug': SUCCESS,
        ':tracker:extractDebugAnnotations': FROM_CACHE,
        ':tracker:extractDeepLinksDebug': FROM_CACHE,
        ':tracker:extractDeepLinksForAarDebug': FROM_CACHE,
        ':tracker:generateDebugAssets': UP_TO_DATE,
        ':tracker:generateDebugRFile': FROM_CACHE,
        ':tracker:generateDebugResValues': FROM_CACHE,
        ':tracker:generateDebugResources': FROM_CACHE,
        ':tracker:javaPreCompileDebug': FROM_CACHE,
        ':tracker:kaptDebugKotlin': SUCCESS,
        ':tracker:kaptGenerateStubsDebugKotlin': FROM_CACHE,
        ':tracker:mergeDebugConsumerProguardFiles': SUCCESS,
        ':tracker:mergeDebugGeneratedProguardFiles': SUCCESS,
        ':tracker:mergeDebugJavaResource': SUCCESS,
        ':tracker:mergeDebugJniLibFolders': SUCCESS,
        ':tracker:mergeDebugNativeLibs': NO_SOURCE,
        ':tracker:mergeDebugShaders': SUCCESS,
        ':tracker:packageDebugAssets': SUCCESS,
        ':tracker:packageDebugResources': FROM_CACHE,
        ':tracker:parseDebugLocalResources': FROM_CACHE,
        ':tracker:preBuild': UP_TO_DATE,
        ':tracker:preDebugBuild': UP_TO_DATE,
        ':tracker:prepareDebugArtProfile': SUCCESS,
        ':tracker:prepareLintJarForPublish': SUCCESS,
        ':tracker:processDebugJavaRes': SUCCESS,
        ':tracker:processDebugManifest': FROM_CACHE,
        ':tracker:stripDebugDebugSymbols': NO_SOURCE,
        ':tracker:syncDebugLibJars': FROM_CACHE,
        ':tracker:writeDebugAarMetadata': SUCCESS,
        ':wearable:assembleDebug': SUCCESS,
        ':wearable:checkDebugAarMetadata': SUCCESS,
        ':wearable:checkDebugDuplicateClasses': SUCCESS,
        ':wearable:checkKotlinGradlePluginConfigurationErrors': SKIPPED,
        ':wearable:compileDebugJavaWithJavac': FROM_CACHE,
        ':wearable:compileDebugKotlin': FROM_CACHE,
        ':wearable:compileDebugShaders': NO_SOURCE,
        ':wearable:compressDebugAssets': FROM_CACHE,
        ':wearable:createDebugApkListingFileRedirect': SUCCESS,
        ':wearable:createDebugCompatibleScreenManifests': SUCCESS,
        ':wearable:desugarDebugFileDependencies': FROM_CACHE,
        ':wearable:dexBuilderDebug': FROM_CACHE,
        ':wearable:extractDeepLinksDebug': FROM_CACHE,
        ':wearable:generateDebugAssets': UP_TO_DATE,
        ':wearable:generateDebugResValues': FROM_CACHE,
        ':wearable:generateDebugResources': FROM_CACHE,
        ':wearable:javaPreCompileDebug': FROM_CACHE,
        ':wearable:kaptDebugKotlin': SKIPPED,
        ':wearable:kaptGenerateStubsDebugKotlin': SKIPPED,
        ':wearable:mapDebugSourceSetPaths': SUCCESS,
        ':wearable:mergeDebugAssets': SUCCESS,
        ':wearable:mergeDebugGlobalSynthetics': FROM_CACHE,
        ':wearable:mergeDebugJavaResource': SUCCESS,
        ':wearable:mergeDebugJniLibFolders': SUCCESS,
        ':wearable:mergeDebugNativeDebugMetadata': NO_SOURCE,
        ':wearable:mergeDebugNativeLibs': NO_SOURCE,
        ':wearable:mergeDebugResources': FROM_CACHE,
        ':wearable:mergeDebugShaders': SUCCESS,
        ':wearable:mergeDebugStartupProfile': SUCCESS,
        ':wearable:mergeExtDexDebug': FROM_CACHE,
        ':wearable:mergeLibDexDebug': FROM_CACHE,
        ':wearable:mergeProjectDexDebug': FROM_CACHE,
        ':wearable:packageDebug': SUCCESS,
        ':wearable:preBuild': UP_TO_DATE,
        ':wearable:preDebugBuild': UP_TO_DATE,
        ':wearable:processDebugJavaRes': SUCCESS,
        ':wearable:processDebugMainManifest': FROM_CACHE,
        ':wearable:processDebugManifest': FROM_CACHE,
        ':wearable:processDebugManifestForPackage': FROM_CACHE,
        ':wearable:processDebugResources': FROM_CACHE,
        ':wearable:stripDebugDebugSymbols': NO_SOURCE,
        ':wearable:validateSigningDebug': SUCCESS,
        ':wearable:writeDebugAppMetadata': SUCCESS,
        ':wearable:writeDebugSigningConfigVersions': SUCCESS,
    ]
}
