/*
 * Copyright 2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.tasks.compile

import org.gradle.util.TestUtil
import spock.lang.Specification

class CompileOptionsTest extends Specification {
    static final TEST_DEBUG_OPTION_MAP = [someDebugOption: 'someDebugOptionValue']
    static final TEST_FORK_OPTION_MAP = [someForkOption: 'someForkOptionValue']

    CompileOptions compileOptions

    def setup()  {
        compileOptions = new CompileOptions(TestUtil.objectFactory())
        compileOptions.debugOptions = [optionMap: {TEST_DEBUG_OPTION_MAP}] as DebugOptions
        compileOptions.forkOptions = [optionMap: {TEST_FORK_OPTION_MAP}] as ForkOptions
    }

    @SuppressWarnings("GrDeprecatedAPIUsage")
    def "default compile options"() {
        expect:
        compileOptions.debug
        compileOptions.failOnError
        compileOptions.warnings

        !compileOptions.deprecation
        !compileOptions.listFiles
        !compileOptions.verbose
        !compileOptions.fork

        compileOptions.compilerArgs.empty
        compileOptions.encoding == null
        compileOptions.bootstrapClasspath == null
        compileOptions.extensionDirs == null

        compileOptions.forkOptions != null
        compileOptions.debugOptions != null
    }

    def "fork"() {
        compileOptions.fork = false
        boolean forkUseCalled = false

        compileOptions.forkOptions = [define: {Map args ->
            forkUseCalled = true
            assert args == TEST_FORK_OPTION_MAP
        }] as ForkOptions

        expect:
        compileOptions.fork(TEST_FORK_OPTION_MAP).is(compileOptions)
        compileOptions.fork
        forkUseCalled
    }

    def "debug"() {
        compileOptions.debug = false
        boolean debugUseCalled = false

        compileOptions.debugOptions = [define: {Map args ->
            debugUseCalled = true
            args == TEST_DEBUG_OPTION_MAP
        }] as DebugOptions

        expect:
        assert compileOptions.debug(TEST_DEBUG_OPTION_MAP).is(compileOptions)
        compileOptions.debug
        debugUseCalled
    }

    def "define"() {
        compileOptions.debug = false
        compileOptions.fork = false
        compileOptions.define(debug: true)

        expect:
        compileOptions.debug
        !compileOptions.fork
    }

    def "converts GStrings to Strings when getting all compiler arguments"() {
        given:
        compileOptions.compilerArgs << "Foo${23}"

        expect:
        compileOptions.allCompilerArgs.contains('Foo23')
    }
}
